#include <winsock2.h> // remember to include ws2_32.lib
#include <iostream.h>
#include <ws2tcpip.h>

// Parameters specific to this ping (I think)
#define DEFAULT_PACKET_SIZE 32
#define MAX_PING_DATA_SIZE 1024
#define MAX_PING_PACKET_SIZE (MAX_PING_DATA_SIZE+sizeof(IPHeader))

// ICMP packet types
#define ICMP_ECHO_REPLY 0
#define ICMP_DEST_UNREACH 3
#define ICMP_TTL_EXPIRE 11
#define ICMP_ECHO_REQUEST 8

// Minimum ICMP packet size, in bytes
#define ICMP_MIN 8

/* The following two structures need to be packed tightly, but unlike
Borland C++, Microsoft C++ does not do this by default.
Neither does LCC.  I'm a great fan of pack(1) myself, though some
people say it makes things slower. */

#pragma pack(push,1)

typedef struct IPHeader_tag // =20 bytes
{
	unsigned h_len:4; // Length of the header in dwords
	unsigned version:4; // Version of IP
	BYTE tos; // Type of service
	USHORT total_len; // Length of the packet in dwords
	USHORT ident; // unique identifier
	USHORT flags; // Flags
	BYTE ttl; // Time to live
	BYTE proto; // Protocol number (TCP, UDP etc)
	USHORT checksum; // IP checksum
	ULONG source_ip;
	ULONG dest_ip;
} IPHeader;

typedef struct ICMPHeader_tag // =8 bytes
{
	BYTE type; // ICMP packet type
	BYTE code; // Type sub code
	USHORT checksum;
	USHORT id;
	USHORT seq;
} ICMPHeader;
#pragma pack(pop)

USHORT ip_checksum(USHORT *buffer,int size)
{
	unsigned long cksum=0;
	while (size>1)
	{
		cksum+=*buffer++;
		size-=sizeof(USHORT);
	}
	if (size) cksum+=*(UCHAR*)buffer;
	cksum=(cksum>>16)+(cksum&0xFFFF);
	cksum+=(cksum>>16);
	return (USHORT)(~cksum);
}

void errr(const char *error)
{
	// Stephen Brooks's Most Excellent Error-Checking Routine
}

int setup_for_ping(unsigned addr,SOCKET& sd,SOCKADDR_IN& dest)
{
	sd=WSASocket(AF_INET,SOCK_RAW,IPPROTO_ICMP,0,0,0);
	if (sd==INVALID_SOCKET)
	{
		char str[1000];
		sprintf(str,"Failed to create raw socket: %08X",WSAGetLastError()); errr(str);
		return -1;
	}
	memset(&dest,0,sizeof(dest));
	if (addr!=INADDR_NONE)
	{
		dest.sin_addr.s_addr=addr;
		dest.sin_family=AF_INET;
	}
	return 0;
}

void init_ping_packet(ICMPHeader *icmp_hdr,int packet_size,int seq_no)
{
	icmp_hdr->type=ICMP_ECHO_REQUEST;
	icmp_hdr->code=0;
	icmp_hdr->checksum=0;
	icmp_hdr->id=(USHORT)GetCurrentProcessId(); // HAHA if I fake this I can kack up Yahoo etc.
	icmp_hdr->seq=seq_no;

	const unsigned long deadmeat=0xDEADBEEF; // Err yes, how amusing
	char *datapart=(char *)icmp_hdr+sizeof(ICMPHeader);
	int bytes_left=packet_size-sizeof(ICMPHeader);
	while (bytes_left>0)
	{
		memcpy(datapart,&deadmeat,min(sizeof(deadmeat),bytes_left));
		bytes_left-=sizeof(deadmeat);
		datapart+=sizeof(deadmeat);
	}
	icmp_hdr->checksum=ip_checksum((USHORT *)icmp_hdr,packet_size);
}

int send_ping(SOCKET sd,const SOCKADDR_IN &dest,ICMPHeader *send_buf,int packet_size)
{
	int bwrote=sendto(sd,(char *)send_buf,packet_size,0,(SOCKADDR *)&dest,sizeof(dest));
	if (bwrote==SOCKET_ERROR)
	{
		char str[1000];
		sprintf(str,"send failed: %08X",WSAGetLastError()); errr(str);
		return -1;
	}
	else if (bwrote<packet_size) printf("sent %d bytes...",bwrote);
	return 0;
}

int recv_ping(SOCKET sd,SOCKADDR_IN& source,IPHeader *recv_buf,int packet_size)
{
	int fromlen=sizeof(source);
	int bread=recvfrom(sd,(char *)recv_buf,packet_size+sizeof(IPHeader),0,(SOCKADDR *)&source,&fromlen);
	if (bread==SOCKET_ERROR)
	{
		char str[1000];
		if (WSAGetLastError()==WSAEMSGSIZE) sprintf(str,"read failed: buffer too small");
		else sprintf(str,"read failed: error #%08X",WSAGetLastError());
		errr(str);
		return -1;
	}
	return 0;
}

int decode_reply(IPHeader *reply,int bytes,SOCKADDR_IN *from,unsigned *ipout)
{
	unsigned short header_len=reply->h_len*4;
	ICMPHeader *icmphdr=(ICMPHeader *)((char *)reply+header_len);
	char str[1000];
	if (bytes<header_len+ICMP_MIN)
	{
		sprintf(str,"too few bytes from %s",inet_ntoa(from->sin_addr)); errr(str);
		return -1;
	}
	else if (icmphdr->type!=ICMP_ECHO_REPLY)
	{
		if (icmphdr->type!=ICMP_TTL_EXPIRE)
		{
			if (icmphdr->type==ICMP_DEST_UNREACH) errr("Destination unreachable");
			else {sprintf(str,"Unknown ICMP packet type %d received",(int)icmphdr->type); errr(str);}
			return -1;
		}
	}
	else if (icmphdr->id!=(USHORT)GetCurrentProcessId()) return -2;
	// ICMP_TTL_EXPIRE, part-way there.  ICMP_ECHO_REPLY at destination.
	*ipout=from->sin_addr.s_addr;
	return 0;
}

#define IP(a,b,c,d) ((unsigned)((d)<<24)|((c)<<16)|((b)<<8)|(a))

unsigned *gettracevector(const unsigned ip,const int maxhops,USHORT& seq_base)
{
	unsigned tstart=GetTickCount(),*ret;
	SOCKET sd; SOCKADDR_IN dest,source;
	if (setup_for_ping(ip,sd,dest)<0) return NULL;
	const int packet_size=32,timeout=500;
	ICMPHeader *send_buf=(ICMPHeader *)malloc(packet_size);
	for (int n=1;n<=maxhops;n++) // Send 'em
	{
		if (setsockopt(sd,IPPROTO_IP,IP_TTL,(char *)&n,sizeof(n))==SOCKET_ERROR)
		{
			char str[1000];
			sprintf(str,"TTL setsockopt failed: WSAGetLastError() = %08X",WSAGetLastError()); errr(str);
			free(send_buf); closesocket(sd); return NULL;
		}
		if (setsockopt(sd,SOL_SOCKET,SO_RCVTIMEO,(char *)&timeout,sizeof(int))==SOCKET_ERROR)
		{
			char str[1000];
			sprintf(str,"TTL setsockopt failed: WSAGetLastError() = %08X",WSAGetLastError()); errr(str);
			free(send_buf); closesocket(sd); return NULL;
		}
		init_ping_packet(send_buf,packet_size,seq_base+n);
		send_ping(sd,dest,send_buf,packet_size);
	}
	free(send_buf);
	IPHeader *recv_buf=(IPHeader *)malloc(MAX_PING_PACKET_SIZE);
	ret=(unsigned *)calloc(maxhops+1,sizeof(unsigned));
	char mank=0; int count=0;
	while (GetTickCount()<tstart+timeout && !mank && count<maxhops) // Receive 'em
	{
		if (recv_ping(sd,source,recv_buf,MAX_PING_PACKET_SIZE)<0) {errr("Generic Error at line 217"); mank=1;}
		ICMPHeader *icmphdr=(ICMPHeader *)((char *)recv_buf+(recv_buf->h_len*4));
		if (icmphdr->type==ICMP_TTL_EXPIRE)
		{
			IPHeader *iph=(IPHeader *)((char *)icmphdr+sizeof(ICMPHeader));
			icmphdr=(ICMPHeader *)((char *)iph+(iph->h_len*4));
		}
		unsigned servip;
		int result=decode_reply(recv_buf,packet_size,&source,&servip);
		if (result==0)
		{
			ret[0]=recv_buf->dest_ip;
			int i=icmphdr->seq-seq_base;
			if (i>=0 && i<=maxhops) {ret[i]=servip; count++;} // Success
		}
		else if (result==-1) mank=1;
	}
	free(recv_buf); closesocket(sd); // Get rid of those pesky pigeons
	seq_base=seq_base/1000*1000+(seq_base%1000+maxhops+1)%(1000-maxhops-1);
	return ret;
}

#include "netdb.c"
#include <tcconio.h>

void statusreport(const int nodes,const int links,const unsigned long nextsave,const unsigned long size)
{
	gotoxy(1,1);
	printf("%9d nodes %9d links   Save in %2d sec  [",
		nodes,links,nonegi(floor(0.001*(int)(nextsave-GetTickCount()))));
	if (size<1024) printf("%d bytes",size);
	else if (size<10240) printf("%.2f KB",(float)size/1024.0);
	else if (size<102400) printf("%.1f KB",(float)size/1024.0);
	else if (size<1048576) printf("%d KB",size/1024);
	else if (size<10485760) printf("%.2f MB",(float)size/1048576.0);
	else if (size<104857600) printf("%.1f MB",(float)size/1048576.0);
	else printf("%d MB",size/1048576);
	printf(" in file]"); clreol();
}

BOOL CALLBACK title_cb(HWND hwnd,LPARAM nodes)
{
	char str[100];
	GetWindowText(hwnd,str,99); strlwr(str);
	if (strstr(str,"netmap"))
	{
		sprintf(str,"Netmap: %d nodes",nodes);
		SetWindowText(hwnd,str);
	}
	return TRUE; // Keep going
}

typedef enum {Random=0,Mutate=1,Crossover=2} TrialType;
#define TRIALTYPES 3

void efficiencyreport(const TrialType t,const int nn)
{
	int score[TRIALTYPES],runs[TRIALTYPES];
	FILE *in=fopen(".\\efficiency.txt","rt"),*out;
	if (!in)
	{
		out=fopen(".\\efficiency.txt","wt");
		for (int n=0;n<TRIALTYPES;n++) fprintf(out,"0,0\n");
		fclose(out);
		in=fopen(".\\efficiency.txt","rt");
	}
	for (int n=0;n<TRIALTYPES;n++) fscanf(in,"%d,%d\n",score+n,runs+n);
	fclose(in);
	score[t]+=nn; runs[t]++;
	out=fopen(".\\efficiency.txt","wt");
	for (int n=0;n<TRIALTYPES;n++) fprintf(out,"%d,%d\n",score[n],runs[n]);
	fclose(out);
}

const char *basefilename=".\\web";
const int steps=50,saveinterval=60;

//#include "thread_driver.c"
#include "normal_driver.c"
