#ifndef SBCLIB_ARRAY_2D
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <rnd/Max.c>

#ifndef REAL
#define REAL double // Assume double precision
#endif

REAL **array_2d(const unsigned xs,const unsigned ys)
{
	REAL **a=(REAL **)malloc((2+xs)*sizeof(REAL *));
	*(unsigned *)a=xs; *(unsigned *)(a+1)=ys; a+=2;
	a[0]=(REAL *)malloc(xs*ys*sizeof(REAL));  // Do this with only a single malloc by putting everything in one block
	for (int x=xs-1;x>=0;x--) a[x]=a[0]+x*ys;
	return a;
}

#define array_2d_xs(A) (*(unsigned *)((A)-2))
#define array_2d_ys(A) (*(unsigned *)((A)-1))

void array_2d_free(REAL **a)
{
	free(a[0]); free(a-2);
}

#define array_2d_type(xs,ys,T) (T **)array_2d_type_new(xs,ys,sizeof(T))
void **array_2d_type_new(const unsigned xs,const unsigned ys,const unsigned sz)
{
	void **a=(void **)malloc((2+xs)*sizeof(void *));
	*(unsigned *)a=xs; *(unsigned *)(a+1)=ys; a+=2;
	a[0]=(void *)malloc(xs*ys*sz);
	for (int x=xs-1;x>=0;x--) a[x]=(unsigned char *)a[0]+x*ys*sz;
	return a;
}
#define array_2d_type_zeroed(xs,ys,T) (T **)array_2d_type_zeroed_new(xs,ys,sizeof(T))
void **array_2d_type_zeroed_new(const unsigned xs,const unsigned ys,const unsigned sz)
{
	void **a=(void **)malloc((2+xs)*sizeof(void *));
	*(unsigned *)a=xs; *(unsigned *)(a+1)=ys; a+=2;
	a[0]=(void *)calloc(xs*ys,sz);
	for (int x=xs-1;x>=0;x--) a[x]=(unsigned char *)a[0]+x*ys*sz;
	return a;
}
#define array_2d_type_free(a) array_2d_free((REAL **)a)
#define array_2d_type_zero(a,T) array_2d_type_zero_inner((void **)(a),sizeof(T))
void array_2d_type_zero_inner(void **a,const unsigned sz)
{
	int xs=array_2d_xs(a),ys=array_2d_ys(a);
	memset(a[0],0x00,xs*ys*sz);
}

REAL **array_2d_zeroed(const unsigned xs,const unsigned ys)
{
	int x,y; REAL **a=(REAL **)malloc((2+xs)*sizeof(REAL *));
	*(unsigned *)a=xs; *(unsigned *)(a+1)=ys; a+=2;
	a[0]=(REAL *)malloc(xs*ys*sizeof(REAL));
	for (x=xs-1;x>=0;x--)
	{
		a[x]=a[0]+x*ys;
		for (y=ys-1;y>=0;y--) a[x][y]=0;
	}
	return a;
}

REAL **array_2d_norows(const unsigned xs,const unsigned ys)
{ // Attach your own rows from e.g. malloced arrays
	REAL **a=(REAL **)malloc((2+xs)*sizeof(REAL *));
	*(unsigned *)a=xs; *(unsigned *)(a+1)=ys; a+=2;
	return a;
}

void array_2d_zero(REAL **a)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]=0;
}

REAL **array_2d_copy(const REAL * const *c)
{
	unsigned xs=array_2d_xs(c),ys=array_2d_ys(c);
	REAL **a=(REAL **)malloc((2+xs)*sizeof(REAL *));
	memcpy(a,c-2,2*sizeof(REAL *)); a+=2;
	a[0]=(REAL *)malloc(xs*ys*sizeof(REAL));
	for (int x=xs-1;x>=0;x--) a[x]=a[0]+x*ys;
	memcpy(a[0],c[0],xs*ys*sizeof(REAL));
	return a;
}

void array_2d_eq(REAL **a,const REAL * const *b)
{ // a and b should be the same size, this just copies the data
	memcpy(a[0],b[0],array_2d_xs(b)*array_2d_ys(b)*sizeof(REAL));
}

void array_2d_scale(REAL **a,const REAL k)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]*=k;
}

void array_2d_addconst(REAL **a,const REAL b)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]+=b;
}

void array_2d_add(REAL **a,const REAL * const *b)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]+=b[x][y];
}

void array_2d_sub(REAL **a,const REAL * const *b)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]-=b[x][y];
}

void array_2d_addsc(REAL **a,const REAL k,REAL * const * const b)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]+=k*b[x][y];
}

void array_2d_mul(REAL **a,const REAL * const *b)
{ // b should be at least as big as a
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]*=b[x][y];
}

REAL array_2d_max(const REAL * const *a)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	REAL ret=a[0][0];
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) if (a[x][y]>ret) ret=a[x][y];
	return ret;
}

REAL array_2d_min(const REAL * const *a)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	REAL ret=a[0][0];
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) if (a[x][y]<ret) ret=a[x][y];
	return ret;
}

REAL array_2d_sum(const REAL * const *a)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	REAL ret=0;
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) ret+=a[x][y];
	return ret;
}

REAL array_2d_mean(const REAL * const *a)
{
	return array_2d_sum(a)/(array_2d_xs(a)*array_2d_ys(a));
}

REAL array_2d_supnorm(const REAL * const *a)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	REAL ret=fabs(a[0][0]);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) ret=Max(ret,fabs(a[x][y]));
	return ret;
}

REAL array_2d_sumsq(const REAL * const *a)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	REAL ret=0;
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) ret+=a[x][y]*a[x][y];
	return ret;
}

REAL array_2d_norm(const REAL * const *a)
{
	return sqrt(array_2d_sumsq(a));
}

REAL array_2d_rms(const REAL * const *a)
{
	return sqrt(array_2d_sumsq(a)/(array_2d_xs(a)*array_2d_ys(a)));
}

REAL **array_2d_transpose(const REAL * const *a)
{
	int x,y,xs,ys; REAL **ret=array_2d(xs=array_2d_ys(a),ys=array_2d_xs(a));
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) ret[x][y]=a[y][x];
	return ret;
}

#if defined(__stdio_h__) || defined(_STDIO_H_) || defined(_STDIO_H) || defined(_INC_STDIO) || defined(_STDIO_DEFINED)
void array_2d_print(const REAL * const *a,const char *fmt="\t%lg")
{
	unsigned x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=0;x<xs;x++)
	{
		for (y=0;y<ys;y++) printf(fmt,a[x][y]);
		printf("\n");
	}
	printf("\n");
}

void array_2d_printwrite(const REAL * const *a,FILE *out,const char transpose=0,const char *nfmt="%lg",const char *sep="\t")
{
	unsigned x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	if (transpose) for (y=0;y<ys;y++) for (x=0;x<xs;x++)
		{fprintf(out,nfmt,a[x][y]); if (x+1==xs) fputs("\n",out); else fputs(sep,out);}
	else for (x=0;x<xs;x++) for (y=0;y<ys;y++)
		{fprintf(out,nfmt,a[x][y]); if (y+1==ys) fputs("\n",out); else fputs(sep,out);}
	fprintf(out,"\n");
}

void array_2d_printfile(const REAL * const *a,const char *filename,const char transpose=0,const char *mode="wt")
{ // transpose included because array[small][large] is RAM efficient but wrong way for Excel
	FILE *out=fopen(filename,mode);
	array_2d_printwrite(a,out,transpose);
	fclose(out);
}

void array_2d_writeCSV(const REAL * const * a,FILE *out,const int transpose=0)
{
	array_2d_printwrite(a,out,transpose,"%.16lg",",");
}

void array_2d_saveCSV(const REAL * const * a,const char *filename,const int transpose=0)
{ // Goes with array_2d_loadCSV
	FILE *out=fopen(filename,"wt");
	array_2d_writeCSV(a,out,transpose);
	fclose(out);
}

void array_2d_write(const REAL * const *a,FILE *out)
{ // Binary
	unsigned x,xs=array_2d_xs(a),ys=array_2d_ys(a);
	fwrite(&xs,sizeof(unsigned),1,out);
	fwrite(&ys,sizeof(unsigned),1,out);
	for (x=0;x<xs;x++) fwrite(a[x],sizeof(REAL),ys,out);
}

void array_2d_save(const REAL * const *a,const char *filename)
{ // Binary
	FILE *out=fopen(filename,"wb");
	array_2d_write(a,out);
	fclose(out);
}

REAL **array_2d_read(FILE *in)
{ // Binary
	unsigned x,xs,ys;
	fread(&xs,sizeof(unsigned),1,in);
	fread(&ys,sizeof(unsigned),1,in);
	REAL **ret=array_2d(xs,ys);
	for (x=0;x<xs;x++) fread(ret[x],sizeof(REAL),ys,in);
	return ret;
}

void array_2d_readinto(FILE *in,REAL **a)
{ // Binary
	unsigned x,xs,ys;
	fread(&xs,sizeof(unsigned),1,in);
	fread(&ys,sizeof(unsigned),1,in);
	for (x=0;x<xs;x++) fread(a[x],sizeof(REAL),ys,in);
}

REAL **array_2d_load(const char *filename)
{ // Binary
	FILE *in=fopen(filename,"rb");
	REAL **ret=array_2d_read(in);
	fclose(in);
	return ret;
}

#include <math.h>
void array_2d_printstructure(const REAL * const *a,const REAL eps=1e-12)
{
	unsigned x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=0;x<xs;x++)
	{
		for (y=0;y<ys;y++)
		{
			if (fabs(a[x][y])<eps) printf(".");
			else if (fabs(a[x][y])*eps>1) printf("*");
			else if (a[x][y]>0) printf("+");
			else if (a[x][y]<0) printf("-");
			else printf("#");
		}
		printf("\n");
	}
	printf("\n");
}
#endif
#ifdef SBCLIB_BITMAPFONTS
void array_2d_display(const int ox,const int oy,const int cs,const REAL * const *a,const unsigned c)
{
	char str[100]; int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=ys-1;x>=0;x--) for (y=xs-1;y>=0;y--)
	{
		sprintf(str,"%lg",a[y][x]);
		bfwrite(ox+x*BF_height*cs,oy+y*BF_height,str,c);
	}
}
#endif
#ifdef SBCLIB_GRAPHICS
#include <rnd/nonegi.c>
void array_2d_plot_rect(const int ox,const int oy,const int zx,const int zy,const REAL * const *a,unsigned colmap(REAL))
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a),
		x0=nonegi(-ox/zx),x1=Mini(xs,(xres-ox)/zx+1),
		y0=nonegi(-oy/zy),y1=Mini(ys,(yres-oy)/zy+1);
	if (x0>=xs || x1<0 || y0>=ys || y1<0) return;
	#ifdef SBCLIB_GRAPHICS_OPENGL
	glBegin(GL_QUADS);
	#endif
	for (x=x0;x<x1;x++) for (y=y0;y<y1;y++)
		#ifdef SBCLIB_GRAPHICS_OPENGL
		rect_inner
		#else
		rect
		#endif
			(ox+x*zx,oy+y*zy,zx,zy,colmap(a[x][y]));
	#ifdef SBCLIB_GRAPHICS_OPENGL
	glEnd();
	#endif
}

void array_2d_plot(const int ox,const int oy,const int z,const REAL * const *a,unsigned colmap(REAL))
	{array_2d_plot_rect(ox,oy,z,z,a,colmap);}
#endif

#include <rnd/Mini.c>
#include <rnd/nonegi.c>
#include <rnd/noneg.c>

REAL array_2d_bilinear(const REAL * const *a,const REAL x,const REAL y)
{ // Not sure if this is the fastest - might be worth trying to optimise a bit some time
	int xs=array_2d_xs(a),ys=array_2d_ys(a),
		ix=Mini((int)noneg(x),xs-1),iy=Mini((int)noneg(y),ys-1),
		jx=Mini(ix+1,xs-1),jy=Mini(iy+1,ys-1);
	REAL fx=x-ix,fy=y-iy;
	return (1.0-fx)*((1.0-fy)*a[ix][iy]+fy*a[ix][jy])+fx*((1.0-fy)*a[jx][iy]+fy*a[jx][jy]);
}

#ifdef SBCLIB_CSV
#include <rnd/Maxi.c>
REAL **array_2d_fromCSV(csv c)
{
	int x,y,xs=csvm(c),ys=csvrowm(c[0]);
	for (x=xs-1;x>0;x--) ys=Maxi(ys,csvrowm(c[x]));
	REAL **ret=array_2d_zeroed(xs,ys);
	for (x=xs-1;x>=0;x--) for (y=Mini(ys,csvrowm(c[x]))-1;y>=0;y--) ret[x][y]=atof(c[x][y]);
	return ret;
}

REAL **array_2d_loadCSV(const char *filename,const int headerrows=0,
	const char *sep=",",const char quotes=0,const char *commentprefix=NULL)
{
	csv c=csvload(filename,sep,quotes,commentprefix);
	if (!c) return NULL;
	if (headerrows) csvdeleteheaderrow(c,headerrows);
	REAL **ret=array_2d_fromCSV(c);
	csvfree(c);
	return ret;
}

REAL **array_2d_loadCSVsome(const char *filename,const int start=0,const int end=-1,const int every=1,
	const int headerrows=0,const char *sep=",",const char quotes=0,const char *commentprefix=NULL)
{
	csv c=csvloadsome(filename,start,end,every,sep,quotes,commentprefix);
	if (!c) return NULL;
	if (headerrows) csvdeleteheaderrow(c,headerrows);
	REAL **ret=array_2d_fromCSV(c);
	csvfree(c);
	return ret;
}
#endif

#define SBCLIB_ARRAY_2D
#endif
