#ifndef SBCLIB_MAGNET_ROTATINGCOIL
// This is like <magnet/rotcoildata.c> except without the complicated structure
#include <magnet/PM2D.c>
#include <magnet/goalfield.c>

void (*Magnet_rotatingcoil_applyerrorset)(REAL *pn,REAL *ps,REAL r)=NULL;

void Magnet_rotatingcoil(const Magnet mag,const V2 c,const REAL r,REAL *pn,REAL *ps,const int m=360)
{ // Fills pn,ps array_1ds (must be same size), units are Teslas at radius r
	int n,i,pm=array_1d_m(pn); V2 p,B,d,pd; REAL ox;
	array_1d_zero(pn); array_1d_zero(ps);
	for (n=m-1;n>=0;n--)
	{
		d=V2_degs(360.0*n/m);
		p=c+r*d;
		B=Magnet_field(mag,p);
		pd=V2_new(0,1);
		for (i=0;i<pm;i++)
		{
			pn[i]+=B.y*pd.y-B.x*pd.x; // Agrees with Animesh's By>0 for x>0 for positive normal pole convention
			ps[i]+=B.x*pd.y+B.y*pd.x; // Agrees with Animesh's Bx>0 for x>0 for positive skew pole convention
			ox=pd.x;
			pd.x=pd.x*d.x-pd.y*d.y;
			pd.y=ox*d.y+pd.y*d.x;
		}
	}
	array_1d_scale(pn,1.0/m); array_1d_scale(ps,1.0/m);
	if (Magnet_rotatingcoil_applyerrorset) Magnet_rotatingcoil_applyerrorset(pn,ps,r);
}

REAL *Magnet_rotatingcoil_interleaved(const Magnet mag,const V2 c,const REAL r,const int pm,const int m=360)
{
	REAL *pn=array_1d(pm),*ps=array_1d(pm),*ret=array_1d(pm*2);
	Magnet_rotatingcoil(mag,c,r,pn,ps);
	for (int n=pm-1;n>=0;n--) {ret[n*2]=pn[n]; ret[n*2+1]=ps[n];}
	array_1d_free(pn); array_1d_free(ps);
	return ret;
}

void rotatingcoil_subtract_GoalField(REAL *pn,REAL *ps,const GoalField goal)
{
	int i,pm=array_1d_m(pn),gm=array_1d_m(goal.norm);
	REAL rp=1;
	for (i=0;i<pm && i<gm;i++) {pn[i]-=goal.norm[i]*rp; ps[i]-=goal.skew[i]*rp; rp*=goal.gfr;}
}

REAL rotatingcoil_ufom(const REAL *pn,const REAL *ps,const GoalField goal,const int refp)
{ // NB: currently these leave main pole in, should use rotatingcoil_subtract_GoalField first
	int i,pm=array_1d_m(pn);
	REAL tuu=0,Bref=goal.norm[refp]*goal.gfr,Bunit=Bref/1e4,u;
	for (i=pm-1;i>=0;i--)
	{
		u=pn[i]/Bunit; tuu+=u*u;
		u=ps[i]/Bunit; tuu+=u*u;
	}
	return sqrt(tuu);
}

REAL rotatingcoil_maxu(const REAL *pn,const REAL *ps,const GoalField goal,const int refp)
{
	int i,pm=array_1d_m(pn);
	REAL maxu=0,Bref=goal.norm[refp]*goal.gfr,Bunit=Bref/1e4,u;
	for (i=pm-1;i>=0;i--)
	{
		u=pn[i]/Bunit; maxu=Max(maxu,fabs(u));
		u=ps[i]/Bunit; maxu=Max(maxu,fabs(u));
	}
	return maxu;
}

void rotatingcoil_toSI(REAL *pn,REAL *ps,const REAL r)
{ // Converts the result from Magnet_rotatingcoil into SI units (T/m^n) independent of coil radius
	if (r<=1e-10) reportfatal("r<=1e-10 in Magnet_rotatingcoil_toSI");
	int i,pm=array_1d_m(pn); REAL k=1;
	for (i=0;i<pm;i++) {pn[i]*=k; ps[i]*=k; k/=r;}
}

#include <rnd/sreplace.c>
#include <rnd/sdreplace.c>

void rotatingcoil_latextable(const REAL *pn,const REAL *ps,const int refp,const REAL r,
	const char *filename,const char *title=NULL)
{ // Like RotCoilData_latextable except without the RotCoilData structure
	FILE *out=fopen(filename,"wt");
	fprintf(out,"\\documentclass{article}\n");
	fprintf(out,"\\usepackage{booktabs}\n");
	fprintf(out,"\\begin{document}\n");
	fprintf(out,"\\begin{table}[htb]\n\\caption[]{Field harmonics");
	if (title)
	{
		char *tr=sdreplace(sreplace(title,"\\","\\textbackslash "),"_","\\_");
		fprintf(out," from %s",tr); free(tr);
	}
	fprintf(out," at R=%lgmm",r/1e-3);
	fprintf(out,".  The reference field component is %lg T",pn[refp]/pow(r,refp));
	if (refp>1) fprintf(out,"/m$^%d$",refp);
	else if (refp==1) fprintf(out,"/m");
	fprintf(out,".}\n");
	fprintf(out,"\\begin{tabular*}{\\columnwidth}{@{\\extracolsep{\\fill}}lcc}\n\\toprule\n");
	fprintf(out,"Field harmonic&Normal units&Skew units\\\\\n");
	fprintf(out,"\\midrule\n");
	const char *polename[]={"Dipole","Quadrupole","Sextupole","Octupole","Decapole","Dodecapole"};
	int pm=array_1d_m(pn); REAL Bunit=pn[refp]/1e4;
	for (int i=0;i<pm;i++)
	{
		if (i<=5) fprintf(out,"%s",polename[i]); else fprintf(out,"%d-pole",2*(1+i));
		fprintf(out,"&%.2lf&%.2lf\\\\\n",pn[i]/Bunit,ps[i]/Bunit);
	}
	fprintf(out,"\\bottomrule\n");
	fprintf(out,"\\end{tabular*}\n\\label{tab:rotatingcoil}\n\\end{table}\n");
	fprintf(out,"\\end{document}");
	fclose(out);
}

#include <csv.c>
#include <array_2d.c>

void rotatingcoil_saveCSV(const REAL *pn,const REAL *ps,const int refp,const REAL r,const char *filename)
{ // Like RotCoilData_saveCSV
	int n,pm=array_1d_m(pn); REAL **f=array_2d(1+pm,2);
	REAL Bref=pn[refp],Bunit=Bref/1e4;
	f[0][0]=Bref/pow(r,refp); f[0][1]=0;
	for (n=0;n<pm;n++) {f[1+n][0]=pn[n]/Bunit; f[1+n][1]=ps[n]/Bunit;}
	array_2d_saveCSV(f,filename);
	array_2d_free(f);
}

#define SBCLIB_MAGNET_ROTATINGCOIL
#endif
