typedef struct
{
	REAL R,Br;
	int wedges, // Initial number of sides of magnet (e.g. 16)
		midplanes, // Number of midplanes to remove: 1=left, 2=horizontal, 4=cross
		halfoff, // Offset segments by half a width: 0=CBETA, 1=Nick's orientation
		removeadjacent, // Remove wedges adjacent to midplane: 0=No, 1=Yes
		symmetry; // Enforce symmetry: 1=none, 2=top/bottom, 4=quadrupole
	REAL ymidplane, // Half-height of open midplane to be removed
		thmidplane; // Half-opening-angle of open midplane
} HalbachOpen;

const HalbachOpen HalbachOpen_default={2.5e-2,1.07, 16,2,-1,-1,2, 0.0,0.0}; // -1 means not set manually, call HalbachOpen_consistency

void HalbachOpen_consistency(HalbachOpen *p,const REAL yoval=0)
{
	if (yoval>0)
	{ // Recommended: halbacharea_maxiters=Maxi(halbacharea_maxiters,50);
		if (yoval>=p->R) reportfatal("yoval should be smaller than horizontal aperture radius.");
		if (p->midplanes!=2) reportfatal("Midplanes must equal 2 with oval magnet.");
		if (p->halfoff>=0) reportwarning("halfoff ignored with oval magnet.");
		if (p->removeadjacent>=0) reportfatal("removeadjacent ignored with oval magnet.");
		if (p->symmetry!=2) reportfatal("Symmetry must equal 2 with oval magnet.");
	}
	if (p->thmidplane>0)
	{
		if (p->halfoff>0) reportwarning("halfoff ignored with thmidplane set.");
		p->halfoff=0;
	}
	if (p->halfoff<0) p->halfoff=(p->ymidplane>0); // Contextual defaults if not set manually and not forced by thmidplane
	if (p->removeadjacent<0) p->removeadjacent=(p->ymidplane>0 && p->thmidplane==0);
}

int makemagnet_halbach_open_params(const HalbachOpen p)
{
	int removedwedges;
	if (p.ymidplane>0) removedwedges=p.removeadjacent*p.midplanes*(p.halfoff?1:2);
	else removedwedges=0;
	return (p.wedges-removedwedges+p.symmetry-1)/p.symmetry*2; // Round up before dividing to allow on-edge ones
}

Magnet makemagnet_halbach_open(const REAL *a,const HalbachOpen p)
{
	Magnet ret=Magnet_new();
	if (p.removeadjacent && p.wedges%p.midplanes)
	{
		char str[300];
		sprintf(str,"removeadjacent=1 but number of wedges (%d) not a multiple of number of midplanes (%d) in makemagnet_halbach_open",
			p.wedges,p.midplanes);
		reportfatala(str);
	}
	if (p.thmidplane!=0 && p.midplanes!=p.symmetry && p.midplanes*2!=p.symmetry)
	{
		char str[300];
		sprintf(str,"makemagnet_halbach_open: With thmidplane, symmetry (%d here) must either be equal to midplanes (%d) or midplanes*2 (%d)",
			p.symmetry,p.midplanes,p.midplanes*2);
		reportfatala(str);		
	}
	if (p.thmidplane!=0 && p.halfoff!=0) reportfatal("makemagnet_halbach_open: With thmidplane, halfoff should be set to zero");
	int params=makemagnet_halbach_open_params(p);
	if (array_1d_m(a)!=params)
	{
		char str[300];
		sprintf(str,"Wrong number of parameters given to makemagnet_halbach_open: got %d, required %d for \
wedges=%d midplanes=%d halfoff=%d removeadjacent=%d symmetry=%d",array_1d_m(a),params,
			p.wedges,p.midplanes,p.halfoff,p.removeadjacent,p.symmetry);
		reportfatala(str);
	}
	int n,i;
	REAL th=(M_TWOPI-p.midplanes*2*p.thmidplane)/p.wedges,hth=0.5*th,hw=p.R*tan(hth),dr,rot;
	PMPoly pmp; V2 v;
	for (n=(p.wedges+p.symmetry-1)/p.symmetry-(!p.halfoff || p.symmetry==1),i=0;n>=0;n--)
	{
		if (p.removeadjacent)
		{
			if ((n+p.wedges/2)%(p.wedges/p.midplanes)==0) continue;
			if (!p.halfoff && (n+p.wedges/2+1)%(p.wedges/p.midplanes)==0) continue;
		}
		v=V2_new(a[i],a[i+1]); i+=2;
		dr=minthick+(maxthick-minthick)*sq(sin(sqrt(V2_norm(v))));
		rot=p.thmidplane+n*th+(!p.halfoff)*hth;
		if (p.ymidplane>0 && (p.midplanes>=2 || n>0))
		{ // Make sure near-midplane-trim pieces aren't lost entirely
			REAL s1=sin(rot-hth),s2=sin(rot+hth),smax=Max(fabs(s1),fabs(s2)),
				rmin=p.ymidplane/smax;
			if (p.midplanes==4)
			{
				REAL c1=cos(rot-hth),c2=cos(rot+hth),cmax=Max(fabs(c1),fabs(c2));
				rmin=Max(rmin,p.ymidplane/cmax);
			}
			rmin*=cos(hth); // Convert to mid-side r
			if (rmin>p.R+minthick) dr+=rmin-(p.R+minthick);
		}
//sprintf(str,"dr[%d] = %lg m",n,dr); bfwrite(xres/4,n*BF_height,str,GFX_white);
		pmp=PMPoly_new();
		Poly_addpoint(&pmp.p,p.R,hw);
		REAL hf=hw*(p.R+dr)/p.R;
		Poly_addpoint(&pmp.p,p.R+dr,hf);
		Poly_addpoint(&pmp.p,p.R+dr,-hf);
		Poly_addpoint(&pmp.p,p.R,-hw);
		PMPoly_rotate(&pmp,rot);
		pmp.Br=p.Br*V2_normalise(v);
		LSet_add(&ret.pmpolys,&pmp);
		if (p.symmetry>=2 && fabs(fmod(rot+0.5*M_PI,M_PI)-0.5*M_PI)>0.001)
		{
			// Add symmetrical one
			{
				PMPoly q=PMPoly_copy(pmp);
				Poly_flipy(&q.p);
				q.Br.x*=-1;
				LSet_add(&ret.pmpolys,&q);
			}
			if (p.symmetry==4 && fabs(fmod(rot+0.25*M_PI,0.5*M_PI)-0.25*M_PI)>0.001)
			{
				PMPoly qq=PMPoly_copy(pmp);
				Poly_flipx(&qq.p);
				qq.Br.y*=-1;
				LSet_add(&ret.pmpolys,&qq);
				// Add symmetrical one
				{
					PMPoly q=PMPoly_copy(qq);
					Poly_flipy(&q.p);
					q.Br.x*=-1;
					LSet_add(&ret.pmpolys,&q);
				}
			}
		}
	}
	Magnet_clip_midplanes(&ret,p.ymidplane,p.midplanes);
	return ret;
}
