#define HALBACHAREA_STANDALONE // If all in one folder

#include "halbacharea.c"
#include <stdio.h>
#include <rnd/explode.c>
#include <rnd/fileexists.c>

void magnet_stats_log(const GoalField goal,const HalbachOpen p,const REAL yoval,const char *filename,
	const Magnet mag,const REAL *pn,const REAL *ps,const int converged,const int refp)
{
	int already=fileexists("magnet_stats.csv");
	FILE *out=fopen("magnet_stats.csv","at");
	if (!already) fprintf(out,"Filename, \
Dipole (T),Quad (T/m),B_r (T),R_aperture (mm),y_oval (mm),y_midplane (mm),th_midplane (degrees),x_goodfield (mm),R_goodfield (mm), \
Midplanes,Halfoff,Removeadjacent,Symmetry, Wedges, \
Area (cm^2),Max radius (cm),Strength error (relative),Max harmonic (units),FOM (units)\n");
	fprintf(out,"%s, %.9lg,%.9lg, %.9lg,%.9lg,%.9lg,%.9lg,%.9lg,%.9lg,%.9lg, %d,%d,%d,%d, %d, ",filename,goal.norm[0],goal.norm[1],
		p.Br,p.R/1e-3,yoval/1e-3,p.ymidplane/1e-3,p.thmidplane radians,goal.gfc.x/1e-3,goal.gfr/1e-3,
		p.midplanes,p.halfoff,p.removeadjacent,p.symmetry,
		p.wedges);
	if (1 || converged)
	{
		REAL area=Magnet_area(mag),maxr=Magnet_maxr(mag);
		fprintf(out,"%.9lg,%.9lg",area/1e-4,maxr/1e-2);
		if (yoval==0) // Harmonics on circle are not valid for oval aperture magnets
		{
			REAL *epn=array_1d_copy(pn),*eps=array_1d_copy(ps);
			rotatingcoil_subtract_GoalField(epn,eps,goal);
			fprintf(out,",%.9lg,%.9lg,%.9lg",
				epn[refp]/(goal.norm[refp]*pow(goal.gfr,refp)),
				rotatingcoil_maxu(epn,eps,goal,refp),
				rotatingcoil_ufom(epn,eps,goal,refp));
			array_1d_free(epn); array_1d_free(eps);
		}
	}
	if (!converged)
	{
		fprintf(out,",Not converged");
	}
	fprintf(out,"\n");
	fclose(out);
}

int main(int argc,const char *argv[])
{
	if (argc<=1) reportfatal("Usage:\nhalbacharea R=30e-3 dipole=0.2 quad=30\n\n\
Other switch examples:\n\
gfr=0.025 good field region radius (m) (default 2R/3)\n\
sext=66, oct=888 set higher poles (T/m^2, T/m^3)\n\
p6=66, p8=888, p4s=123 set arbitrary multipole or skew multipole\n\
Br=1.3 set remnant field (T) (default 1.07)\n\
wedges=24 set initial number of angular wedges (default 16)\n\
symmetry=1(none),2(top/bottom),4(quad) set symmetry type (default 2)\n\
halfoff=1 offset segments by half a width (default 1 if open midplane, else 0)\n\
ymidplane=0.002 open midplane half-height (m) (default 0), enables:\n\
  midplanes=1(left),2(horiz),4(cross) number of midplanes to remove (default 2)\n\
  removeadjacent=1 remove wedges adjacent to open midplane (default 1)\n\
thmidplane=10 open midplane half-angle (degrees)\n\
yoval=0.01 set oval aperture half-height (m) (default off)\n\
clearance=50e-6 set full gap between adjacent blocks\n\
iters=100 change number of iterations (default 25 or 50)\n\
name=ABC set filename base (default 'magnet')");
	int n,i,mpole=1; REAL gfr=0;
	HalbachOpen p=HalbachOpen_default;
	REAL yoval=0,clearance=0; // Try oval if yoval!=0
	char str[200],name[200]; strcpy(name,"magnet");
	for (n=1;n<argc;n++)
	{
		char **e=explode("=",argv[n]);
		if (!strcmpi(e[0],"quad")) mpole=Maxi(mpole,2);
		else if (!strcmpi(e[0],"sext")) mpole=Maxi(mpole,3);
		else if (!strcmpi(e[0],"oct")) mpole=Maxi(mpole,4);
		else if (!strcmpi(e[0],"dec")) mpole=Maxi(mpole,5);
		else if (!strcmpi(e[0],"dodec")) mpole=Maxi(mpole,6);
		else if (lc(e[0][0])=='p' && isdigit(e[0][1])) // p6, p4s etc.
		{
			i=atoi(e[0]+1);
			if (i%2) reportfatal("Odd number of poles specified on multipole component!");
			if (i<2) reportfatal("Number of poles out of range (too low, below dipole)");
			i=i/2-1; // Now array index
			mpole=Maxi(mpole,i+1);
		}
		else if (!strcmpi(e[0],"gfr")) gfr=atof(e[1]);
		else if (!strcmpi(e[0],"R")) p.R=atof(e[1]);
		else if (!strcmpi(e[0],"Br")) p.Br=atof(e[1]);
		else if (!strcmpi(e[0],"wedges")) p.wedges=atoi(e[1]);
		else if (!strcmpi(e[0],"midplanes")) p.midplanes=atoi(e[1]);
		else if (!strcmpi(e[0],"halfoff")) p.halfoff=atoi(e[1]);
		else if (!strcmpi(e[0],"removeadjacent")) p.removeadjacent=atoi(e[1]);
		else if (!strcmpi(e[0],"symmetry")) p.symmetry=atoi(e[1]);
		else if (!strcmpi(e[0],"ymidplane")) p.ymidplane=atof(e[1]);
		else if (!strcmpi(e[0],"thmidplane")) p.thmidplane=atof(e[1]) degrees;
		else if (!strcmpi(e[0],"yoval")) yoval=atof(e[1]);
		else if (!strcmpi(e[0],"clearance")) clearance=atof(e[1]);
		else if (!strcmpi(e[0],"iters")) halbacharea_maxiters=atoi(e[1]);
		else if (!strcmpi(e[0],"name")) strcpy(name,e[1]);
		else if (!strcmpi(e[0],"length")) halbacharea_save_length=atof(e[1]);
		explodefree(e);
	}
	HalbachOpen_consistency(&p,yoval);
	if (yoval>0) halbacharea_maxiters=Maxi(halbacharea_maxiters,50);
	if (gfr==0) gfr=p.R*2.0/3;
	GoalField goal=GoalField_zero(mpole);
	for (n=1;n<argc;n++)
	{
		char **e=explode("=",argv[n]);
		if (!strcmpi(e[0],"dipole")) goal.norm[0]=atof(e[1]);
		else if (!strcmpi(e[0],"quad")) goal.norm[1]=atof(e[1]);
		else if (!strcmpi(e[0],"sext")) goal.norm[2]=atof(e[1]);
		else if (!strcmpi(e[0],"oct")) goal.norm[3]=atof(e[1]);
		else if (!strcmpi(e[0],"dec")) goal.norm[4]=atof(e[1]);
		else if (!strcmpi(e[0],"dodec")) goal.norm[5]=atof(e[1]);
		else if (lc(e[0][0])=='p' && isdigit(e[0][1])) // p6, p4s etc.
		{
			i=atoi(e[0]+1);
			i=i/2-1; // Now array index
			if (lc(e[0][strlen(e[0])-1])=='s')
			{
				if (p.symmetry>1) reportwarning("Typically magnets with skew components require symmetry=1 to be set");
				goal.skew[i]=atof(e[1]);
			}
			else goal.norm[i]=atof(e[1]);
		}
		explodefree(e);
	}
	int refp=1; for (n=0;n<mpole;n++) if (goal.norm[n]!=0) {refp=n; break;}
	if (mpole>1 && refp==0 && goal.norm[1]!=0) refp=1;
	goal.gfr=gfr;
	int pm=halbacharea_maxpole; REAL *pn=array_1d(pm),*ps=array_1d(pm);
	Magnet mag;
	sprintf(str,"%s.csv",name);
	int ret=halbacharea(goal,p, str, &mag,pn,ps, yoval,clearance);
	magnet_stats_log(goal,p,yoval,str,mag,pn,ps,ret,refp);
	if (yoval==0) // Harmonics on circle are not valid for oval aperture magnets
	{
		sprintf(str,"%s_harmonics.tex",name);
		rotatingcoil_latextable(pn,ps,refp,gfr,str);
		sprintf(str,"%s_harmonics.csv",name);
		rotatingcoil_saveCSV(pn,ps,refp,gfr,str);
	}
	array_1d_free(pn); array_1d_free(ps);
	GoalField_free(goal);
	REAL area=Magnet_area(mag),maxr=Magnet_maxr(mag);
	Magnet_free(mag);
	if (!ret) reportfatal("Not converged");
	printf("\nArea=%lg cm^2\nMax radius=%lg cm\n",area/1e-4,maxr/1e-2);
}
