#ifndef SBCLIB_ARRAY_1D

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <rnd/Mini.c>
#include <rnd/Max.c>

#ifndef REAL
#define REAL double // Assume double precision
#endif

REAL *array_1d(const unsigned s)
{
	REAL *a=(REAL *)malloc((1+s)*sizeof(REAL));
	*(unsigned *)a=s; // Assumes REAL is at least as long as unsigned
	return a+1;
}

#define array_1d_m(A) (*(unsigned *)(((REAL *)A)-1))

void array_1d_free(REAL *a)
{
	free(a-1);
}

#define array_1d_type(s,T) (T *)array_1d_type_new(s,sizeof(T))
void *array_1d_type_new(const unsigned s,const int sz)
{
	unsigned char *a=(unsigned char *)malloc(sizeof(REAL)+s*sz);
	*(unsigned *)a=s; // Assumes REAL is at least as long as unsigned
	return a+sizeof(REAL);
}
#define array_1d_type_free(a) array_1d_free((REAL *)a)
void *array_1d_type_copy_inner(const REAL *c,const int sz)
{ // Constructor
	unsigned m=array_1d_m(c);
	unsigned char *a=(unsigned char *)malloc(sizeof(REAL)+m*sz);
	memcpy(a,c-1,sizeof(REAL)+m*sz);
	return a+sizeof(REAL);
}
#define array_1d_type_copy(a,T) (T *)array_1d_type_copy_inner(a,sizeof(T))

REAL *array_1d_zeroed(const unsigned s)
{
	REAL *a=(REAL *)malloc((1+s)*sizeof(REAL));
	*(unsigned *)a=s; // Assumes REAL is at least as long as unsigned
	for (unsigned n=s;n>0;n--) a[n]=0;
	return a+1;
}

#include <rnd/explode.c>

REAL *array_1d_fromCSV(const char *s)
{ // Constructor from a string like "1.0,0.5,0.33"
	char **e=explode(",",s);
	int n,m=explodesegs(e); REAL *ret=array_1d(m);
	for (n=m-1;n>=0;n--) ret[n]=atof(e[n]);
	explodefree(e);
	return ret;
}

void array_1d_zero(REAL *a)
{
	int n,m=array_1d_m(a);
	for (n=m-1;n>=0;n--) a[n]=0;
}

void array_1d_fill(REAL *a,const REAL x)
{
	int n,m=array_1d_m(a);
	for (n=m-1;n>=0;n--) a[n]=x;
}

REAL *array_1d_copy(const REAL *c)
{ // Constructor
	unsigned m=array_1d_m(c); REAL *a=(REAL *)malloc((1+m)*sizeof(REAL));
	memcpy(a,c-1,(1+m)*sizeof(REAL));
	return a+1;
}

void array_1d_eq(REAL *a,const REAL *b)
{
	memcpy(a,b,Mini(array_1d_m(a),array_1d_m(b))*sizeof(REAL));
}

void array_1d_extendby(REAL **pa,const unsigned x)
{
	unsigned m=array_1d_m(*pa); REAL *a=(REAL *)malloc((1+m+x)*sizeof(REAL));
	*(unsigned *)a=m+x;
	memcpy(a+1,*pa,m*sizeof(REAL));
	free(*pa-1); *pa=a+1;
}

void array_1d_extendzeroed(REAL **pa,const unsigned nm)
{
	unsigned m=array_1d_m(*pa);
	if (m>=nm) return;
	REAL *a=(REAL *)malloc((1+nm)*sizeof(REAL));
	*(unsigned *)a=nm;
	memcpy(a+1,*pa,m*sizeof(REAL));
	for (unsigned n=nm;n>m;n--) a[n]=0;
	free(*pa-1); *pa=a+1;
}

#if defined(__stdio_h__) || defined(_STDIO_H_) || defined(_STDIO_H) || defined(_INC_STDIO) || defined(_STDIO_DEFINED)
void array_1d_print(const REAL *a,const char *nfmt="%lg",const char *sep=" ",const char *begin="[ ",const char *end=" ]\n")
{
	unsigned n,m=array_1d_m(a);
	printf("%s",begin);
	for (n=0;n<m;n++) {printf(nfmt,a[n]); if (n+1<m) printf("%s",sep);}
	printf("%s",end);
}

void array_1d_printwrite(const REAL *a,FILE *out,const char *nfmt="%lg",const char *sep="\t",const char *begin="",const char *end="\n")
{
	unsigned n,m=array_1d_m(a);
	fputs(begin,out);
	for (n=0;n<m;n++) {fprintf(out,nfmt,a[n]); if (n+1<m) fputs(sep,out);}
	fputs(end,out);
}

void array_1d_printfile(const REAL *a,const char *filename,const char *nfmt="%lg",const char *sep="\t",const char *begin="",const char *end="\n")
{
	FILE *out=fopen(filename,"at");
	array_1d_printwrite(a,out,nfmt,sep,begin,end);
	fclose(out);
}

void array_1d_writeCSVcol(const REAL *a,FILE *out,const char *nfmt="%.16lg")
	{array_1d_printwrite(a,out,nfmt,"\n");}

void array_1d_writeCSVrow(const REAL *a,FILE *out,const char *nfmt="%.16lg")
	{array_1d_printwrite(a,out,nfmt,",");}

void array_1d_saveCSVcol(const REAL *a,const char *filename,const char *nfmt="%.16lg")
	{array_1d_printfile(a,filename,nfmt,"\n");}

void array_1d_saveCSVrow(const REAL *a,const char *filename,const char *nfmt="%.16lg")
	{array_1d_printfile(a,filename,nfmt,",");}

void array_1d_save(const REAL *a,const char *filename)
{ // Faster binary format
	FILE *out=fopen(filename,"wb");
	unsigned m=array_1d_m(a);
	fwrite(&m,1,sizeof(unsigned),out);
	fwrite(a,m,sizeof(REAL),out);
	fclose(out);
}

REAL *array_1d_load(const char *filename)
{
	FILE *in=fopen(filename,"rb");
	unsigned m;
	fread(&m,1,sizeof(unsigned),in);
	REAL *ret=array_1d(m);
	fread(ret,m,sizeof(REAL),in);
	fclose(in);
	return ret;
}
#endif

void array_1d_scale(REAL *a,const REAL k)
{
	for (int n=array_1d_m(a)-1;n>=0;n--) a[n]*=k;
}

void array_1d_addconst(REAL *a,const REAL k)
{
	for (int n=array_1d_m(a)-1;n>=0;n--) a[n]+=k;
}

void array_1d_add(REAL *a,const REAL *b)
{ // Make sure b is at least as big as a
	for (int n=array_1d_m(a)-1;n>=0;n--) a[n]+=b[n];
}

void array_1d_sub(REAL *a,const REAL *b)
{ // Make sure b is at least as big as a
	for (int n=array_1d_m(a)-1;n>=0;n--) a[n]-=b[n];
}

void array_1d_addsc(REAL *a,const REAL k,const REAL *b)
{ // Make sure b is at least as big as a
	for (int n=array_1d_m(a)-1;n>=0;n--) a[n]+=k*b[n];
}

void array_1d_mul(REAL *a,const REAL *b)
{ // Make sure b is at least as big as a
	for (int n=array_1d_m(a)-1;n>=0;n--) a[n]*=b[n];
}

REAL array_1d_max(const REAL *a)
{
	REAL ret=a[0];
	for (int n=array_1d_m(a)-1;n>0;n--) if (a[n]>ret) ret=a[n];
	return ret;
}

REAL array_1d_min(const REAL *a)
{
	REAL ret=a[0];
	for (int n=array_1d_m(a)-1;n>0;n--) if (a[n]<ret) ret=a[n];
	return ret;
}

REAL array_1d_sum(const REAL *a)
{
	REAL ret=0;
	for (int n=array_1d_m(a)-1;n>=0;n--) ret+=a[n];
	return ret;
}

REAL array_1d_sumabs(const REAL *a)
{ // Sum of absolute values aka L1 norm
	REAL ret=0;
	for (int n=array_1d_m(a)-1;n>=0;n--) ret+=fabs(a[n]);
	return ret;
}

REAL array_1d_mean(const REAL *a)
{
	return array_1d_sum(a)/array_1d_m(a);
}

REAL array_1d_supnorm(const REAL *a)
{ // L_infinity norm
	REAL ret=fabs(a[0]);
	for (int n=array_1d_m(a)-1;n>0;n--) ret=Max(ret,fabs(a[n]));
	return ret;
}

REAL array_1d_sumsq(const REAL *a)
{
	REAL ret=0;
	for (int n=array_1d_m(a)-1;n>=0;n--) ret+=a[n]*a[n];
	return ret;
}

REAL array_1d_norm(const REAL *a)
{ // L2 norm
	return sqrt(array_1d_sumsq(a));
}

REAL array_1d_rms(const REAL *a)
{
	return sqrt(array_1d_sumsq(a)/array_1d_m(a));
}

REAL array_1d_dot(const REAL *a,const REAL *b)
{
	REAL ret=0;
	for (int n=Mini(array_1d_m(a),array_1d_m(b))-1;n>=0;n--) ret+=a[n]*b[n];
	return ret;
}

REAL array_1d_dd(const REAL *a,const REAL *b)
{
	REAL ret=0;
	for (int n=Mini(array_1d_m(a),array_1d_m(b))-1;n>=0;n--) ret+=(a[n]-b[n])*(a[n]-b[n]);
	return ret;
}

REAL array_1d_dist(const REAL *a,const REAL *b)
{
	return sqrt(array_1d_dd(a,b));
}

#include <rnd/qsort_type.c>
QSORT(REAL)

void array_1d_qsort(REAL *a)
{
	qsort_REAL(a,array_1d_m(a));
}

REAL array_1d_linear(const REAL *a,const REAL x)
{ // Linear interpolation
	if (x<=0) return a[0];
	int m=array_1d_m(a);
	if (x>=m-1) return a[m-1];
	int n=(int)x; REAL f=x-n;
	return (1.0-f)*a[n]+f*a[n+1];
}

#ifdef SBCLIB_BITMAPFONTS
void array_1d_display(const int ox,const int oy,const REAL *a,const unsigned c)
{
	char str[100]; int n,m=array_1d_m(a);
	for (n=m-1;n>=0;n--)
	{
		sprintf(str,"%lg",a[n]);
		bfwrite(ox,oy+n*BF_height,str,c);
	}
}
#endif
#ifdef SBCLIB_GRAPHICS
void array_1d_plot(const int ox,const int oy,const int z,const REAL *a,unsigned colmap(REAL))
{ // Draws a column vector analogous with array_2d_plot
	#ifdef SBCLIB_GRAPHICS_OPENGL
	glBegin(GL_QUADS);
	#endif
	int n,m=array_1d_m(a);
	for (n=m-1;n>=0;n--)
		#ifdef SBCLIB_GRAPHICS_OPENGL
		rect_inner
		#else
		rect
		#endif
			(ox,oy+n*z,z,z,colmap(a[n]));
	#ifdef SBCLIB_GRAPHICS_OPENGL
	glEnd();
	#endif
}
#endif

#define SBCLIB_ARRAY_1D
#endif
