#ifndef SBCLIB_REPORT
// Stack of task statuses [Stephen Brooks 2016]
//- <report.c> replacement for "textio.c" and <profile[1].c>?
//#include <profile1.c> // see also Muon1 "textio.c"
#ifdef NOREPORT
#define report(str)
#define reporta(str)
#define reportbegin(str)
#define reportbegina(str)
#define reportend()
#define reportfatal(str)
#define reportfatala(str)
#define reportwarning(str)
#define reportwarninga(str)
#define reportprogress(A)
#define reportprogressd(A,B)
#define reportprogressinc(A)
#else

#include <lset.c>
#include <rnd/eithertime.c>
#include <rnd/streqi.c>
#include <rnd/lc.c>

typedef enum
{
	ReportType_null=0,
	ReportInfo=1,ReportBegin=2,ReportEnd=3,
	ReportProgress=4,
	ReportFatal=5, // Alerts the user, terminates the program
	ReportWarning=6, // Alerts the user, can wait for "OK"
	ReportYesNo=7
} ReportType;
// More types: [WARNING], [FATAL] (error) queries from Muon1, questions Y/N, multi-choice or user input

typedef struct
{
	ReportType type;
	const char *file; int line; unsigned thread; // For file assume static string allocated elsewhere
	char *title; int title_alloced;
	double tstart,progress,progress_denom;
	int answer;
} ReportEntry;

const ReportEntry ReportEntry_null={ReportType_null, NULL,0,0, NULL,0, 0,0,0, 0};

void ReportEntry_free(ReportEntry *e) {if (e->title_alloced) free(e->title);}

LSet report_stack; char report_init=0; double report_init_time;
void report_display_default(ReportEntry *);
void (*report_display)(ReportEntry *)=report_display_default; // Could be set by user before report_initialise

void report_initialise(void)
{
	if (report_init) return;
	report_stack=LSet(ReportEntry);
	LSet_add(&report_stack,&ReportEntry_null);
	report_init_time=eithertime();
	report_init=1;
}

#define report(str) report_inner(__FILE__,__LINE__,ReportInfo,str)
#define reporta(str) report_inner(__FILE__,__LINE__,ReportInfo,str,1)
#define reportbegin(str) report_inner(__FILE__,__LINE__,ReportBegin,str)
#define reportbegina(str) report_inner(__FILE__,__LINE__,ReportBegin,str,1)
#define reportend() report_inner(__FILE__,__LINE__,ReportEnd)
#define reportfatal(str) report_inner(__FILE__,__LINE__,ReportFatal,str)
#define reportfatala(str) report_inner(__FILE__,__LINE__,ReportFatal,str,1)
#define reportwarning(str) report_inner(__FILE__,__LINE__,ReportWarning,str)
#define reportwarninga(str) report_inner(__FILE__,__LINE__,ReportWarning,str,1)
#define reportyesno(str) report_inner(__FILE__,__LINE__,ReportYesNo,str)
#define reportyesnoa(str) report_inner(__FILE__,__LINE__,ReportYesNo,str,1)

ReportEntry *report_stack_top(void)
{
	return ((ReportEntry *)report_stack.a)+(report_stack.m-1);
}

void report_stack_pop(void)
{
	ReportEntry_free(report_stack_top());
	LSet_remove(&report_stack,report_stack.m-1);
}

int report_inner(const char *file,const int line,const ReportType t=ReportInfo,const char *str=NULL,
	const int alloc=0)
{
	report_initialise();
	if (t==ReportEnd)
	{
		if (report_stack.m>1) report_stack_pop();
		else reportfatal("Trying to pop empty stack in report_inner");
		if (report_display)
		{
			report_stack_top()->type=ReportEnd;
			report_display(report_stack_top());
		}
		ReportEntry_free(report_stack_top());
		*report_stack_top()=ReportEntry_null;
	}
	else if (t==ReportBegin || t==ReportInfo || t==ReportFatal || t==ReportWarning || t==ReportYesNo)
	{
		ReportEntry e; e.type=t;
		e.file=file; e.line=line;
		#ifdef _WINDOWS_
		e.thread=GetCurrentThreadId();
		#else
		e.thread=0;
		#endif
		if (alloc) {streqi(&e.title,str); e.title_alloced=1;}
		else {e.title=(char *)str; e.title_alloced=0;}
		e.tstart=eithertime();
		e.progress=e.progress_denom=0;
		e.answer=0;
		ReportEntry_free(report_stack_top());
		int temporary=(t==ReportWarning || t==ReportYesNo);
		if (temporary) LSet_add(&report_stack,&e); // Temporarily add to top
		else *report_stack_top()=e; // Replace top
		if (report_display) report_display(report_stack_top());
		int ret=report_stack_top()->answer;
		if (t==ReportBegin) LSet_add(&report_stack,&ReportEntry_null);
		else if (t==ReportFatal) exit(1);
		if (temporary) report_stack_pop();
		return ret;
	}
	else reportfatal("Unknown ReportType in report_inner");
	return 0;
}

ReportEntry *report_stack_progress_top(void)
{
	ReportEntry *ret=report_stack_top();
	if (report_stack.m>1 && !ret->title) ret--; // Progress on the previous logically declared thing
	ret->type=ReportProgress;
	return ret;
}

void reportprogress(const double progress)
{
	report_initialise();
	ReportEntry *e=report_stack_progress_top();
	e->progress=progress;
	if (report_display) report_display(e);
}

void reportprogressd(const double progress,const double progress_denom)
{
	report_initialise();
	ReportEntry *e=report_stack_progress_top();
	e->progress=progress; e->progress_denom=progress_denom;
	if (report_display) report_display(e);
}

void reportprogressinc(const double progressinc)
{
	report_initialise();
	ReportEntry *e=report_stack_progress_top();
	e->progress+=progressinc;
	if (report_display) report_display(e);
}

// Outputs:
// - progress as text or graphics
// - optional profiling as saved file or graphics
// - time estimates
// ? possibly abort to diagnostic at given time index

#include <conio.c>
#include <rnd/sjbsleep.c>
#include <rnd/sjbdatetime.c>

int report_skip_warnings=0,report_log_warnings=1;

int report_display_default_warnings(ReportEntry * const e)
{
	if (e->type==ReportFatal)
	{
		#ifdef SBCLIB_GRAPHICS_OPENGL
		void gerror(const char *message);
		if (gl_windowed==1) gerror(e->title);
		else
		#endif
		printf("\n[FATAL ERROR] %s\n",e->title); sjbsleep(2.5);
		return 1;
	}
	else if (e->type==ReportWarning)
	{
		if (report_log_warnings)
		{
			FILE *out=fopen("warnings.log","at"); fprintf(out,"%s %s\n",sjbdatetime(),e->title); fclose(out);
		}
		#ifdef SBCLIB_GRAPHICS_OPENGL
		void gwarn(const char *message);
		if (gl_windowed==1) if (!report_skip_warnings) gwarn(e->title);
		else
		#endif
		{printf("\n[WARNING] %s\nPress any key to continue program\n",e->title); if (!report_skip_warnings) getchar();}
		return 1;
	}
	else return 0;
}

int report_display_default_questions(ReportEntry * const e)
{
	if (e->type==ReportYesNo)
	{
		#ifdef SBCLIB_GRAPHICS_OPENGL
		/**/ // Get from Muon1?
		#endif
		printf("%s (Y/N) ",e->title);
		char str[1000]; scanf("%s",str); e->answer=(lc(*str)=='y');
		return 1;
	}
	else return 0;
}

void report_display_default(ReportEntry * const e)
{ // Text, general purpose
	if (!report_display_default_warnings(e) && !report_display_default_questions(e))
	{
		if (e->title) printf("%s",e->title);
		if (e->progress>0 || e->progress_denom>0)
		{
			if (e->progress_denom>0) printf(" [%lg/%lg]",e->progress,e->progress_denom);
			else printf(" [%lg]",e->progress);
		}
		printf("\n");
	}
}

void report_display_concise(ReportEntry * const e)
{ // Concise (overwrites one line), rate-limited
	if (!report_display_default_warnings(e) && !report_display_default_questions(e))
	{
		static double tnext=0; double tnow=eithertime();
		if (tnow<tnext) return; // Rate limiter
		tnext=tnow+0.02;
		printf("\r");
		ReportEntry *a=(ReportEntry *)report_stack.a;
		for (int n=0;n<report_stack.m;n++)
		{
			if (a[n].title) printf("%s%s",(n?" ":""),a[n].title);
			if (a[n].progress>0 || a[n].progress_denom>0)
			{
				if (a[n].progress_denom>0) printf(" [%lg/%lg]",a[n].progress,a[n].progress_denom);
				else printf(" [%lg]",a[n].progress);
	//int test=(int)((double)(eithertime()-tstart)*(r->ts.m-n)/Maxi(1,n));
	//printf("RModel_csgsplit n=%d / %d (%.3f%%) [%d:%02d:%02d]\n",n,r->ts.m,100.0*n/r->ts.m,test/3600,test/60%60,test%60);
	//int test=(int)((double)(eithertime()-tstart)*(n+1)/Maxi(1,om-1-n));
	//printf("RModel_csgfilter n=%d (%.3f%%) [%d:%02d:%02d]\n",n,100.0*(om-1-n)/om,test/3600,test/60%60,test%60);
			}
		}
	}
	clreol();
}

void report_display_logall(ReportEntry * const e)
{ // Text for now, show everything
	report_display_default_warnings(e); report_display_default_questions(e);
	if (e->type==ReportEnd)
	{ // Time relative to stored time
		double tnow=eithertime();
		printf("[%lf](%lf)",tnow-report_init_time,tnow-e->tstart);
	}
	else if (e->type==ReportProgress) printf("[%lf]",eithertime()-report_init_time);
	else printf("[%lf]",e->tstart-report_init_time);
	printf(" %s:%d{%u} ",e->file,e->line,e->thread);
	switch (e->type)
	{
		case ReportInfo: printf("%s\n",e->title); break;
		case ReportBegin: printf("BEGIN %s\n",e->title); break;
		case ReportEnd: printf("END %s\n",e->title); break;
		case ReportProgress:
			if (e->progress_denom>0) printf("%s [%lg/%lg]\n",e->title,e->progress,e->progress_denom);
			else printf("%s [%lg]\n",e->title,e->progress);
			break;
		case ReportFatal: printf("[FATAL ERROR] %s\n",e->title); break;
		case ReportWarning: printf("[WARNING] %s\n",e->title); break;
		case ReportYesNo: printf("%s? %s\n",e->title,e->answer?"Yes":"No"); break;
	}
}

#endif

#define SBCLIB_REPORT
#endif
