#ifndef SBCLIB_LSET
// LSet = "Literal Set" library for C [Stephen Brooks 2001-16]
// It's literal because the objects are stored in a contiguous block of memory
#include <rnd/macros.c>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <rnd/Maxu.c>
#include <rnd/varnum.c> // For stable binary interface in files

typedef struct
{
	void *a,**hook; // The array and access hook
	unsigned m; // Number of members
	unsigned buffer,minbuffer,elementsize;
	float minpercentusage;
} LSet; // Should be 28 bytes

const LSet LSet_null={NULL,NULL, 0, 0,0,0, 60.0};

#define LSet(type) LSet_new(sizeof(type))
#define LSet_alloc(type,N) LSet_newalloc(sizeof(type),N,0)
#define LSet_allocspace(type,N) LSet_newalloc(sizeof(type),N,1)

int LSet_initalloc=1;

LSet LSet_new(const unsigned s)
{
	LSet ret;
	ret.m=0;
	ret.buffer=ret.minbuffer=LSet_initalloc;
	ret.elementsize=s;
	ret.a=mallocex(ret.buffer*ret.elementsize,"LSet buffer");
	ret.minpercentusage=60;
	ret.hook=NULL;
	return ret;
}

LSet LSet_newalloc(const unsigned s,const unsigned m,const unsigned space=0)
{ // Override LSet_initalloc for just this one set; if space is set, elements are already "there" to be overwritten
	LSet ret;
	ret.m=space?m:0;
	ret.buffer=ret.minbuffer=m;
	ret.elementsize=s;
	ret.a=mallocex(ret.buffer*ret.elementsize,"LSet buffer");
	ret.minpercentusage=60;
	ret.hook=NULL;
	return ret;
}

LSet LSet_copy(const LSet *s)
{
	LSet ret;
	ret=*s;
	ret.a=(void *)mallocex(ret.buffer*ret.elementsize,"LSet buffer from duplication");
	memcpy(ret.a,s->a,ret.buffer*ret.elementsize);
	ret.hook=NULL;
	return ret;
}

void LSet_fullwrite(const LSet *s,FILE *out)
{ // "Full" save, includes memory options and allows large records
	fwritenum(s->m,out);
	fwritenum(s->elementsize,out);
	fwritenum(s->minbuffer,out);
	fwrite(&s->minpercentusage,sizeof(float),1,out);
	fwrite(s->a,s->elementsize,s->m,out);
} // Header = 7-19 bytes

void LSet_write(const LSet *s,FILE *out)
{ // Condensed format for sets with records shorter than 256 bytes
	fwritenum(s->m,out);
	fwritenum(s->elementsize,out);
	fwrite(s->a,s->elementsize,s->m,out);
} // Saves 5-9 fewer bytes per LSet (header = 2-10 bytes)

void LSet_save(LSet *s,const char *filename)
{
	FILE *out=fopen(filename,"wb");
	LSet_write(s,out);
	fclose(out);
}

unsigned char *LSet_fullsavebuf(LSet *s,unsigned *psz)
{
	unsigned char *ret=(unsigned char *)malloc(3*5+sizeof(float)+s->m*s->elementsize),*d=ret;
	writenum(s->m,&d);
	writenum(s->elementsize,&d);
	writenum(s->minbuffer,&d);
	memcpy(d,&s->minpercentusage,sizeof(float)); d+=sizeof(float);
	memcpy(d,s->a,s->m*s->elementsize); d+=s->m*s->elementsize;
	*psz=d-ret;
	return ret;
}

unsigned char *LSet_savebuf(LSet *s,unsigned *psz)
{
	unsigned char *ret=(unsigned char *)malloc(2*5+s->m*s->elementsize),*d=ret;
	writenum(s->m,&d);
	writenum(s->elementsize,&d);
	memcpy(d,s->a,s->m*s->elementsize); d+=s->m*s->elementsize;
	*psz=d-ret;
	return ret;
}

LSet LSet_fullread(FILE *in)
{
	LSet ret;
	ret.m=freadnum(in);
	ret.elementsize=freadnum(in);
	ret.minbuffer=freadnum(in);
	fread(&ret.minpercentusage,sizeof(float),1,in);
	ret.buffer=Maxu(ret.minbuffer,(int)(ret.m/(0.5+0.005*ret.minpercentusage)));
	ret.a=malloc(ret.buffer*ret.elementsize);
	fread(ret.a,ret.elementsize,ret.m,in);
	ret.hook=NULL;
	return ret;
}

LSet LSet_read(FILE *in)
{
	LSet ret;
	ret.m=freadnum(in);
	ret.elementsize=freadnum(in);
	ret.minbuffer=LSet_initalloc;
	ret.minpercentusage=60;
	ret.buffer=Maxu(ret.minbuffer,(int)(ret.m/(0.5+0.005*ret.minpercentusage)));
	ret.a=malloc(ret.buffer*ret.elementsize);
	fread(ret.a,ret.elementsize,ret.m,in);
	ret.hook=NULL;
	return ret;
}

LSet LSet_load(const char *filename)
{
	FILE *in=fopen(filename,"rb");
	LSet ret=LSet_read(in);
	fclose(in);
	return ret;
}

LSet LSet_fullloadbuf(unsigned char *d,unsigned char **pe=NULL)
{
	LSet ret;
	ret.m=readnum(&d);
	ret.elementsize=readnum(&d);
	ret.minbuffer=readnum(&d);
	ret.minpercentusage=*(float *)d; d+=sizeof(float);
	ret.buffer=Maxu(ret.minbuffer,(int)(ret.m/(0.5+0.005*ret.minpercentusage)));
	ret.a=malloc(ret.buffer*ret.elementsize);
	memcpy(ret.a,d,ret.m*ret.elementsize);
	if (pe) *pe=d+ret.m*ret.elementsize;
	ret.hook=NULL;
	return ret;
}

LSet LSet_loadbuf(unsigned char *d,unsigned char **pe=NULL)
{
	LSet ret;
	ret.m=readnum(&d);
	ret.elementsize=readnum(&d);
	ret.minbuffer=LSet_initalloc;
	ret.minpercentusage=60;
	ret.buffer=Maxu(ret.minbuffer,(int)(ret.m/(0.5+0.005*ret.minpercentusage)));
	ret.a=malloc(ret.buffer*ret.elementsize);
	memcpy(ret.a,d,ret.m*ret.elementsize);
	if (pe) *pe=d+ret.m*ret.elementsize;
	ret.hook=NULL;
	return ret;
}

void **LSet_hookup(LSet *s,void **h)
{ // To unhook, set s.hook=NULL or call this with h=NULL
	void **oh=s->hook;
	s->hook=h;
	if (h) *(s->hook)=s->a;
	return oh; // This returns the old hook: later, call LSet_hookup(s,oldhook) to return to previous state
}

unsigned LSet_add(LSet *s,const void *x)
{ // Equivalent to memcpy(LSet_addspace(s),x,s->elementsize);
	unsigned m=s->m,sz=s->elementsize;
	if (m>=s->buffer)
	{
		void *na;
		s->buffer=(int)((m+1.0)/(0.5+0.005*s->minpercentusage));
		na=mallocex(s->buffer*sz,"LSet buffer");
		memcpy(na,s->a,m*sz); free(s->a); s->a=na;
		if (s->hook) *(s->hook)=na;
	}
	memcpy((unsigned char *)s->a+m*sz,x,sz); s->m++;
	return m;
}

void *LSet_addspace(LSet *s,const unsigned n=1)
{ // Add an uninitialised entry (or n entries) and return a pointer to it
	unsigned m=s->m,sz=s->elementsize;
	if (m+n>s->buffer)
	{
		void *na;
		s->buffer=(int)((float)(m+n)/(0.5+0.005*s->minpercentusage));
		na=mallocex(s->buffer*sz,"LSet buffer");
		memcpy(na,s->a,m*sz); free(s->a); s->a=na;
		if (s->hook) *(s->hook)=na;
	}
	s->m+=n;
	return (unsigned char *)s->a+m*sz;
}

unsigned LSet_addmulti(LSet *s,const void *x,const unsigned n)
{
	unsigned m=s->m;
	memcpy(LSet_addspace(s,n),x,n*s->elementsize);
	return m;
}

unsigned LSet_addrepeat(LSet *s,const void *x,const unsigned n)
{
	unsigned m=s->m;
	for (int i=n;i>0;i--) LSet_add(s,x); // Not the fastest I guess
	return m;
}

void LSet_merge(LSet *s,const LSet *t)
{ // s and t must have the same element size
	memcpy(LSet_addspace(s,t->m),t->a,t->m*t->elementsize);
}

void LSet_addshift(LSet *s,const void *x,const unsigned n)
{ // Inserts before entry n, keeps entries in order but it's slower because more has to be moved
	unsigned m=s->m,sz=s->elementsize;
	if (m>=s->buffer)
	{
		void *na;
		s->buffer=(int)((m+1.0)/(0.5+0.005*s->minpercentusage));
		na=mallocex(s->buffer*sz,"LSet buffer");
		memcpy(na,s->a,m*sz); free(s->a); s->a=na;
		if (s->hook) *(s->hook)=na;
	}
	for (unsigned i=m;i>n;i--) memcpy((unsigned char *)s->a+i*sz,(unsigned char *)s->a+(i-1)*sz,sz);
	memcpy((unsigned char *)s->a+n*sz,x,sz); s->m++;
}

int LSet_indexof(const LSet *s,const void *x)
{
	int n;
	for (n=s->m-1;n>=0;n--) if (!memcmp((unsigned char *)s->a+n*s->elementsize,x,s->elementsize)) return n;
	return -1;
}

#define LSet_contains(s,x) (LSet_indexof(s,x)>=0)

int LSet_addunique(LSet *s,const void *x)
{ // Not fast at all (O(N^2)) if used repeatedly (alternative: LSet_add repeatedly and then LSet_removerepeats)
	if (!LSet_contains(s,x)) return LSet_add(s,x);
	else return -1;
}

void LSet_trim(LSet *s)
{ // Resizes LSet so there is no wasted space, use when finished adding things
	unsigned m=s->m,sz=s->elementsize;
	void *na=mallocex(Maxu(1,m*sz),"LSet buffer");
	s->buffer=m;
	if (m>0) memcpy(na,s->a,m*sz); free(s->a); s->a=na;
	if (s->hook) *(s->hook)=na;
}

void LSet_remove(LSet *s,const unsigned n)
{
	unsigned m=s->m-1,sz=s->elementsize;
	if (n>m) return;
	if (n<m) memcpy((unsigned char *)s->a+n*sz,(unsigned char *)s->a+m*sz,sz); s->m--;
	if ((float)m/s->buffer<0.01*s->minpercentusage && s->buffer>s->minbuffer)
	{
		void *na;
		s->buffer=Maxu(s->minbuffer,(int)((float)m/(0.5+0.005*s->minpercentusage)));
		na=mallocex(s->buffer*sz,"LSet buffer");
		memcpy(na,s->a,m*sz); free(s->a); s->a=na;
		if (s->hook) *(s->hook)=na;
	}
}

void LSet_removeshift(LSet *s,unsigned n)
{ // Keeps entries in order but it's slower because more has to be moved
	unsigned m=s->m-1,sz=s->elementsize;
	if (n>m) return;
	else s->m--;
	for (;n<m;n++) memcpy((unsigned char *)s->a+n*sz,(unsigned char *)s->a+(n+1)*sz,sz);
	if ((float)m/s->buffer<0.01*s->minpercentusage && s->buffer>s->minbuffer)
	{
		void *na;
		s->buffer=Maxu(s->minbuffer,(int)((float)m/(0.5+0.005*s->minpercentusage)));
		na=mallocex(s->buffer*sz,"LSet buffer");
		memcpy(na,s->a,m*sz); free(s->a); s->a=na;
		if (s->hook) *(s->hook)=na;
	}
}

void LSet_delete(LSet *s,const void *x)
{
	int i=LSet_indexof(s,x);
	if (i>=0) LSet_remove(s,i);
}

void LSet_deleteshift(LSet *s,const void *x)
{
	int i=LSet_indexof(s,x);
	if (i>=0) LSet_removeshift(s,i);
}

void LSet_clear(LSet *s)
{
	free(s->a);
	s->m=0; s->buffer=s->minbuffer;
	s->a=mallocex(s->buffer*s->elementsize,"LSet buffer (just cleared)");
	if (s->hook) *(s->hook)=s->a;
}

void LSet_freeref(LSet *s)
{
	if (s->hook) *(s->hook)=NULL;
	free(s->a); free(s);
}

void LSet_free(LSet *s)
{
	if (s->hook) *(s->hook)=NULL;
	free(s->a);
}

typedef int (*sortfn)(const void *,const void *);

void LSet_qsort(LSet *s,const sortfn cmp)
{ // cmp compares pointers to elements of the LSet
	qsort(s->a,s->m,s->elementsize,cmp);
}

unsigned LSet_sortfn_default_sz;
int LSet_sortfn_default(const void *a,const void *b) {return memcmp(a,b,LSet_sortfn_default_sz);}

void LSet_qsort_default(LSet *s)
{ // Sorts by binary content (memcmp) - doesn't work with integers due to byte order!
	LSet_sortfn_default_sz=s->elementsize;
	qsort(s->a,s->m,s->elementsize,LSet_sortfn_default);
}

void LSet_removerepeats(LSet *s)
{
	int n,sz=s->elementsize;
	LSet_qsort_default(s);
	for (n=s->m-1;n>0;n--)
		if (!memcmp((unsigned char *)s->a+n*sz,(unsigned char *)s->a+(n-1)*sz,sz))
			LSet_remove(s,n);
}

LSet LSet_union(const LSet *s,const LSet *t)
{
	LSet ret=LSet_copy(s); LSet_merge(&ret,t); LSet_removerepeats(&ret);
	return ret;
}

LSet LSet_intersect(const LSet *s,const LSet *t)
{ // This assumes the original sets had no repeats
	int n,sz=s->elementsize;
	LSet ret=LSet_copy(s); LSet_merge(&ret,t); LSet_qsort_default(&ret);
	for (n=ret.m-1;n>0;n--)
		if (memcmp((unsigned char *)ret.a+n*sz,(unsigned char *)ret.a+(n-1)*sz,sz))
			LSet_remove(&ret,n);
	LSet_remove(&ret,0);
	return ret;
}

LSet LSet_segment(const LSet *s,const int start,int end=-1)
{
	if (end<0) end=s->m;
	LSet ret=LSet_newalloc(s->elementsize,end-start,1);
	memcpy(ret.a,(unsigned char *)s->a+start*s->elementsize,(end-start)*s->elementsize);
	return ret;
}

#include <rnd/rnd.c>

void LSet_randomise(LSet *s)
{
	int n,i,sz=s->elementsize; void *tmp=malloc(sz); unsigned char *a=(unsigned char *)s->a;
	for (n=s->m-1;n>0;n--)
	{
		i=rnd(n+1);
		if (i!=n) {memcpy(tmp,a+n*sz,sz); memcpy(a+n*sz,a+i*sz,sz); memcpy(a+i*sz,tmp,sz);}
	}
	free(tmp);
}

unsigned LSet_memusage(const LSet *s) {return s->buffer*s->elementsize;} // Just indirect data, excluding sizeof(LSet)

// A "neater" interface using macros - creates unforeseen variable names though
#define LSArray(type,name) LSet name##LSet=LSet_new(sizeof(type)); type *name; LSet_hookup(&name##LSet,&name);
#define LSArray_elements(name) name##LSet.m
#define LSArray_add(name,thing) LSet_add(&name##LSet,&thing)
#define LSArray_remove(name,index) LSet_remove(&name##LSet,index)
#define LSArray_free(name) LSet_free(name##LSet)

#define SBCLIB_LSET
#endif
