#ifndef SBCLIB_MAGNET_PM2D
#define REAL double
#include <stdio.h>
#include <array_1d.c>
#include <poly.c>
#include <report.c>
#include <rnd/sq.c>
#include <rnd/Min.c>
#include <muon1/constants.c>
#include <csv.c>
#include <rnd/striseg.c>
#include <rnd/strnulleqi.c>

V2 unitwirefield(const V2 r)
{ // r = x_sample-x_wire; output is for 1 Amp
	return (2e-7/(r.x*r.x+r.y*r.y))*V2_new(-r.y,r.x);
}

V2 unitsheetfield(const V2 r,const REAL a)
{ // r = x_sample-x0_sheet; sheet extends from (0,0) to (a,0), a>0; output is for 1 Amp/metre
	return 2e-7*V2_new(
		-atan(r.x/r.y)+atan((r.x-a)/r.y),
		0.5*(log(r.x*r.x+r.y*r.y)-log(sq(r.x-a)+r.y*r.y)));
}

V2 unitsheetfieldv(const V2 r,const REAL b)
{ // r = x_sample-x0_sheet; sheet extends from (0,0) to (0,b), b>0; output is for 1 Amp/metre
	return 2e-7*V2_new(
		0.5*(log(r.x*r.x+sq(r.y-b))-log(r.x*r.x+r.y*r.y)),
		atan(r.y/r.x)-atan((r.y-b)/r.x));
}

V2 pmblockfield(const V2 r,const V2 a,const V2 Br)
{ // PMBlock extends from (0,0) to a; Br is remnant field vector
	return (-Br.x/mu0)*unitsheetfield(r,a.x)+
		(Br.x/mu0)*unitsheetfield(r-V2_new(0,a.y),a.x)+
		(Br.y/mu0)*unitsheetfieldv(r,a.y)+
		(-Br.y/mu0)*unitsheetfieldv(r-V2_new(a.x,0),a.y);
}

V2 unitsheetfieldvec(const V2 r,const V2 a)
{ // r = x_sample-x0_sheet; sheet extends from (0,0) to a; output is for 1 Amp/metre
	REAL l=V2_norm(a),k=1.0/l;
	V2 f=unitsheetfield(k*V2_new(r.x*a.x+r.y*a.y,-r.x*a.y+r.y*a.x),l);
	return k*V2_new(f.x*a.x-f.y*a.y,f.x*a.y+f.y*a.x);
}

V2 unitwirefieldrep(const V2 r,const REAL a)
{ // 1 Amp cuwires repeated at (na,0) for integer n
	REAL thx=M_TWOPI*r.x/a,thy=M_TWOPI*r.y/a;
	return ((2e-7*M_PI/a)/(cos(thx)-cosh(thy)))*V2_new(sinh(thy),sin(thx));
}

double currentblockindef(const double x,const double y)
{ // Indefinite double integral of y/(x*x+y*y)
	return y*atan(x/y) + 0.5*x*log(x*x+y*y);
	//return (M_PI*x - 2*x*(0.5*M_PI-atan(y/x)) + y*log(x*x+y*y))/2;
}

typedef struct {V2 o,s,Br;} PMBlock;
typedef struct {V2 Br; Poly p; char *name;} PMPoly; // vs has clockwise winding
typedef struct {V2 c,B; REAL r,p;} IronRod; // p=proportion filled from 0 to 1
typedef struct {V2 c; REAL r,I;} CurrentWire; // I is total current in Amps (out of the screen)
typedef struct {V2 o,s; REAL I;} CurrentBlock; // I is total current in Amps (out of the screen)

V2 PMBlock_field(const PMBlock b,const V2 r) {return pmblockfield(r-b.o,b.s,b.Br);}

int PMBlock_inside(const PMBlock b,V2 p) {p-=b.o; return p.x>0 && p.x<b.s.x && p.y>0 && p.y<b.s.y;}

V2 PMBlock_centre(const PMBlock b) {return b.o+0.5*b.s;}

void PMBlock_flipx(PMBlock *pb)
{ // Does not change B_r
	pb->o.x=-pb->o.x-pb->s.x;
}

void PMBlock_flipy(PMBlock *pb)
{ // Does not change B_r
	pb->o.y=-pb->o.y-pb->s.y;
}

REAL PMBlock_area(const PMBlock b) {return b.s.x*b.s.y;}

REAL PMBlocks_area(const LSet bs)
{
	REAL ret=0; PMBlock *b=(PMBlock *)bs.a;
	for (int n=bs.m-1;n>=0;n--) ret+=PMBlock_area(b[n]);
	return ret;
}

PMPoly PMPoly_new(void) {PMPoly ret; ret.Br=V2_zero; ret.p=Poly_new(); ret.name=NULL; return ret;}

PMPoly PMPoly_copy(const PMPoly p)
{
	PMPoly ret; ret.Br=p.Br; ret.p=Poly_copy(p.p); strnulleqi(&ret.name,p.name); return ret;
}

void PMPoly_free(PMPoly p) {Poly_free(&p.p); if (p.name) free(p.name);}

PMPoly PMPoly_newblock(const V2 o,const V2 s,const V2 Br)
{ // Creates a PMBlock as a PMPoly, so rotatable
	PMPoly ret;
	ret.p=Poly_rect(o,s); 
	ret.Br=Br;
	ret.name=NULL;
	return ret;
}

PMPoly PMPoly_newblock(const PMBlock b) {return PMPoly_newblock(b.o,b.s,b.Br);}

void PMPolys_free(LSet *s)
{
	for (int n=s->m-1;n>=0;n--) PMPoly_free(((PMPoly *)s->a)[n]);
	LSet_free(s);
}

void PMPolys_clear(LSet *s)
{
	for (int n=s->m-1;n>=0;n--) PMPoly_free(((PMPoly *)s->a)[n]);
	LSet_clear(s);
}

int PMPolys_collide(const LSet ps)
{
	int n,i,m=ps.m; PMPoly *p=(PMPoly *)ps.a;
	for (n=m-1;n>=0;n--) for (i=n-1;i>=0;i--)
		if (Poly_collide(p[n].p,p[i].p)) return 1;
	return 0;
}

V2 PMPoly_field(const PMPoly p,const V2 r)
{
	int n,m=p.p.m; V2 ret=V2_zero; V2 *v=(V2 *)p.p.a,e;
	for (n=m-1;n>=0;n--)
	{
		e=v[(n+1)%m]-v[n];
		ret+=((p.Br.x*e.x+p.Br.y*e.y)/(mu0*V2_norm(e)))*unitsheetfieldvec(r-v[n],e);
	}
	return ret;
}

V2 PMPoly_force(const PMPoly p,const PMPoly q,const int steps=100)
{ // Outputs force exerted on p by q in Newtons/(metre depth)
	int n,i,m=p.p.m; V2 ret=V2_zero; V2 *v=(V2 *)p.p.a,e,b;
	for (n=m-1;n>=0;n--)
	{
		e=v[(n+1)%m]-v[n];
		b=V2_zero;
		for (i=steps-1;i>=0;i--) b+=PMPoly_field(q,v[n]+((0.5+i)/steps)*e);
		b/=steps;
		ret+=((p.Br.x*e.x+p.Br.y*e.y)/mu0)*V2_rotateright(b,-1);
	}
	return ret;	
}

void PMPoly_rotate(PMPoly *pp,const REAL th,const V2 about=V2_zero)
{ // Also rotates B_r
	Poly_rotate(&pp->p,th,about);
	pp->Br=V2_rotate(pp->Br,th);
}

void PMPoly_rotatec(PMPoly *pp,const REAL th)
{ // Rotates about centre
	PMPoly_rotate(pp,th,Poly_centre(pp->p));
}

void PMPoly_flipx(PMPoly *pp)
{
	Poly_flipx(&pp->p);
	pp->Br.x*=-1;
}

REAL PMPolys_area(const LSet ps)
{
	REAL ret=0; PMPoly *p=(PMPoly *)ps.a;
	for (int n=ps.m-1;n>=0;n--) ret+=Poly_area(p[n].p);
	return ret;
}

void PMPoly_upright(PMPoly *pp,const int side)
{ // Base side on X axis centred (in X) at (0,0)
	Poly_permute(&pp->p,side);
	V2 *v=(V2 *)pp->p.a,
		s=v[1]-v[0],
		c,a;
	PMPoly_rotate(pp,-atan2(s.y,s.x));
	c=Poly_centre(pp->p);
	a=Poly_min(pp->p);
	Poly_move(&pp->p,-V2_new(c.x,a.y));
}

void Poly_openmidplane(Poly *p,const REAL y)
{ // Moves by +/-y away from midplane
	V2 c=Poly_centre(*p);
	if (c.y>0) Poly_move(p,V2_new(0,y));
	else if (c.y<0) Poly_move(p,V2_new(0,-y));
}

void Poly_avoidmidplane(Poly *p,const REAL y)
{ // Pushes corners vertically away from midplane
	int n,m=p->m; V2 *v=(V2 *)p->a,c=Poly_centre(*p);
	for (n=m-1;n>=0;n--) if (fabs(v[n].y)<y) v[n].y=(c.y>0?y:-y);
}

void Poly_avoidvgap(Poly *p,const REAL y)
{ // Pushes corners vertically away from midplane left-hand (x<0) side
	int n,m=p->m; V2 *v=(V2 *)p->a,c=Poly_centre(*p);
	for (n=m-1;n>=0;n--) if (v[n].x<0 && fabs(v[n].y)<y) v[n].y=(c.y>0?y:-y);
}

IronRod IronRod_new(const V2 c,const REAL r,const REAL p=1)
	{IronRod ret; ret.c=c; ret.B=V2_zero; ret.r=r; ret.p=p; return ret;}

V2 dipole_fieldfv(const double x,const double y,const double Bx,const double By,const double r)
{ // Bx*fieldfx+By*fieldfy, calculated a bit faster
    double rr=(x*x+y*y)/(r*r);
    if (rr>=1) return V2_new(Bx*(x*x-y*y)+By*2*x*y,Bx*2*x*y+By*(y*y-x*x))/(rr*rr*r*r);
	else return V2_new(Bx,By);
}

V2 IronRod_field(const IronRod w,const V2 x)
{
	if (w.r<=0 || w.p==0 || w.B==V2_zero) return V2_zero;
	V2 p=x-w.c;
	return w.p*dipole_fieldfv(p.x,p.y,w.B.x,w.B.y,w.r);
}

REAL IronRod_area(const IronRod w) {return M_PI*w.r*w.r*w.p;}

V2 CurrentWire_field(const CurrentWire w,const V2 x)
{ // Valid for 2D "infinite in Z" case.  Also valid for field average of finite length wire, if multiplied by (zlengthwire/nomlengthmagnet)
	if (w.I==0) return V2_zero;
	V2 p=x-w.c; REAL pp=p.x*p.x+p.y*p.y;
	if (pp>=w.r*w.r) return (2e-7*w.I/pp)*V2_new(-p.y,p.x);
	else return (2e-7*w.I/(w.r*w.r))*V2_new(-p.y,p.x);
}

REAL CurrentWire_area(const CurrentWire w) {return M_PI*w.r*w.r;}

V2 CurrentBlock_field(const CurrentBlock b,const V2 x)
{ // Originally from magnet/VFFA_2D_field/vffa_2d_field.c
	V2 ret,a1=x-b.o,a2=a1-b.s;
	ret.x=-(currentblockindef(a1.x,a1.y)-currentblockindef(a2.x,a1.y)-currentblockindef(a1.x,a2.y)+currentblockindef(a2.x,a2.y));
	ret.y=currentblockindef(a1.y,a1.x)-currentblockindef(a1.y,a2.x)-currentblockindef(a2.y,a1.x)+currentblockindef(a2.y,a2.x);
	return ret*(b.I/(b.s.x*b.s.y)*mu0*0.5/M_PI);
}

REAL CurrentBlock_area(const CurrentBlock b) {return b.s.x*b.s.y;}

int CurrentBlock_inside(const CurrentBlock b,V2 p) {p-=b.o; return p.x>0 && p.x<b.s.x && p.y>0 && p.y<b.s.y;}

// The Magnet structure

typedef struct
{
	LSet pmblocks,pmpolys,irons,cuwires,cublocks;
	REAL xiron; // Iron plates at X=+/-xiron if non-zero
} Magnet;

Magnet Magnet_new(void)
{
	Magnet ret;
	ret.pmblocks=LSet(PMBlock);
	ret.pmpolys=LSet(PMPoly);
	ret.irons=LSet(IronRod);
	ret.cuwires=LSet(CurrentWire);
	ret.cublocks=LSet(CurrentBlock);
	ret.xiron=0;
	return ret;
}

void Magnet_free(Magnet mag)
{
	LSet_free(&mag.pmblocks);
	PMPolys_free(&mag.pmpolys);
	LSet_free(&mag.irons);
	LSet_free(&mag.cuwires);	
	LSet_free(&mag.cublocks);
}

void Magnet_clear(Magnet *pm)
{
	LSet_clear(&pm->pmblocks);
	PMPolys_clear(&pm->pmpolys);
	LSet_clear(&pm->irons);
	LSet_clear(&pm->cuwires);	
	LSet_clear(&pm->cublocks);
}

Magnet Magnet_copy(const Magnet mag)
{
	Magnet ret;
	ret.pmblocks=LSet_copy(&mag.pmblocks);
	ret.pmpolys=LSet(PMPoly);
	PMPoly *p=(PMPoly *)mag.pmpolys.a,q;
	for (int n=0;n<mag.pmpolys.m;n++)
	{
		q=PMPoly_copy(p[n]);
		LSet_add(&ret.pmpolys,&q);
	}
	ret.irons=LSet_copy(&mag.irons);
	ret.cuwires=LSet_copy(&mag.cuwires);
	ret.cublocks=LSet_copy(&mag.cublocks);
	ret.xiron=mag.xiron;
	return ret;	
}

void Magnet_merge(Magnet *pm,const Magnet mag)
{
	LSet_merge(&pm->pmblocks,&mag.pmblocks);
	PMPoly *p=(PMPoly *)mag.pmpolys.a,q;
	for (int n=0;n<mag.pmpolys.m;n++)
	{
		q=PMPoly_copy(p[n]);
		LSet_add(&pm->pmpolys,&q);
	}
	LSet_merge(&pm->irons,&mag.irons);
	LSet_merge(&pm->cuwires,&mag.cuwires);
	LSet_merge(&pm->cublocks,&mag.cublocks);
}

/*
int coilcollision(const PMBlock b,const int exclude)
{ // Set exclude=-1 to do all
	int n,rep=0; BLock *a=(PMBlock *)pmblocks.a;
	for (n=pmblocks.m-1;n>=0;n--) if (n!=exclude && c.r2>a[n].r1 && c.r1<a[n].r2) // R overlap
	{
		if (repeats>1) rep=Maxi(1-repeats,Mini(repeats-1,(int)floor(0.5*(c.z1+c.z2-(a[n].z1+a[n].z2))/repeatcell+0.5))); // floor here because (int) chops negative numbers
		if (c.z2>a[n].z1+rep*repeatcell && c.z1<a[n].z2+rep*repeatcell) return 1;
	}
	return 0;
}*/

V2 Magnet_field(const Magnet mag,const V2 p,const int useirons=-1,const int useplates=1,REAL wirecutp=-1)
{ // Returns B vector
	int n; V2 ret=V2_zero;
	//if (PMPolys_collide(&mag.pmpolys)) return ret;
	PMBlock *b=(PMBlock *)mag.pmblocks.a; 
	for (n=mag.pmblocks.m-1;n>=0;n--) ret+=PMBlock_field(b[n],p);
	PMPoly *q=(PMPoly *)mag.pmpolys.a;
	for (n=mag.pmpolys.m-1;n>=0;n--) ret+=PMPoly_field(q[n],p);
	if (useirons)
	{
		IronRod *w=(IronRod *)mag.irons.a;
		if (wirecutp>=0)
		{ // Limits the model to a proportion of the wire below wirecutp
			IronRod wc; if (wirecutp<1e-9) wirecutp=1e-9;
			for (n=mag.irons.m-1;n>=0;n--) if (n+1!=useirons) {wc=w[n]; wc.p=Min(wc.p,wirecutp)/wirecutp; ret+=IronRod_field(wc,p);}
		}
		else for (n=mag.irons.m-1;n>=0;n--) if (n+1!=useirons) ret+=IronRod_field(w[n],p);
	}
	CurrentWire *w=(CurrentWire *)mag.cuwires.a;
	for (n=mag.cuwires.m-1;n>=0;n--) ret+=CurrentWire_field(w[n],p);
	CurrentBlock *cb=(CurrentBlock *)mag.cublocks.a;
	for (n=mag.cublocks.m-1;n>=0;n--) ret+=CurrentBlock_field(cb[n],p);
	if (mag.xiron!=0 && useplates)
	{
		int m=10; V2 f;
		for (n=-m;n<=m;n++) if (n!=0)
		{
			f=Magnet_field(mag,V2_new(n*mag.xiron*2+((n&1)?-p.x:p.x),p.y),useirons,0);
			ret+=V2_new(f.x,((n&1)?-1:1)*f.y);
		}
	}
	return ret;
}

V2 Magnet_gradient(const Magnet mag,const V2 p)
{ // dB/dx is sufficient (dB/dy can be got from 2D free-space Maxwell equations)
	REAL e=1e-6; V2 q,Bx,Bxn;
	q=p; q.x+=e; Bx=Magnet_field(mag,q);
	q=p; q.x-=e; Bxn=Magnet_field(mag,q);
	return (Bx-Bxn)/(e+e);
}

V2 Magnet_wireforce(const Magnet mag,const V2 p)
{ // grad(|B|^2)
	V2 B=Magnet_field(mag,p),g=Magnet_gradient(mag,p);
	return 2.0*V2_new(B*g,B.x*g.y-B.y*g.x);
}

V2 Magnet_magnetisation(const Magnet mag,const V2 p)
{ // Returns M vector
	int n; V2 ret=V2_zero;
	PMBlock *b=(PMBlock *)mag.pmblocks.a; 
	for (n=mag.pmblocks.m-1;n>=0;n--) if (PMBlock_inside(b[n],p)) ret+=b[n].Br;
	PMPoly *q=(PMPoly *)mag.pmpolys.a;
	for (n=mag.pmpolys.m-1;n>=0;n--) if (Poly_inside(q[n].p,p)) ret+=q[n].Br;
	return ret;
}

int Magnet_inside(const Magnet mag,const V2 p)
{
	int n;
	PMBlock *b=(PMBlock *)mag.pmblocks.a; 
	for (n=mag.pmblocks.m-1;n>=0;n--) if (PMBlock_inside(b[n],p)) return 1;
	PMPoly *q=(PMPoly *)mag.pmpolys.a;
	for (n=mag.pmpolys.m-1;n>=0;n--) if (Poly_inside(q[n].p,p)) return 2;
	IronRod *w=(IronRod *)mag.irons.a;
	for (n=mag.irons.m-1;n>=0;n--) if (V2_normsq(w[n].c-p)<w[n].r*w[n].r) return 3;
	CurrentWire *o=(CurrentWire *)mag.cuwires.a;
	for (n=mag.cuwires.m-1;n>=0;n--) if (V2_normsq(o[n].c-p)<o[n].r*o[n].r) return 4;
	CurrentBlock *cb=(CurrentBlock *)mag.cublocks.a;
	for (n=0;n<mag.cublocks.m;n++) if (CurrentBlock_inside(cb[n],p)) return 5;
	return 0;
}

REAL Magnet_area(const Magnet mag)
{
	REAL ret=PMBlocks_area(mag.pmblocks)+PMPolys_area(mag.pmpolys);
	int n; IronRod *w=(IronRod *)mag.irons.a;
	for (n=0;n<mag.irons.m;n++) ret+=IronRod_area(w[n]);
	CurrentWire *cw=(CurrentWire *)mag.cuwires.a;
	for (n=0;n<mag.cuwires.m;n++) ret+=CurrentWire_area(cw[n]);
	CurrentBlock *cb=(CurrentBlock *)mag.cublocks.a;
	for (n=0;n<mag.cublocks.m;n++) ret+=CurrentBlock_area(cb[n]);
	return ret;
}

void Magnet_move(Magnet *pm,const V2 d)
{
	int n; PMBlock *b=(PMBlock *)pm->pmblocks.a;
	for (n=pm->pmblocks.m-1;n>=0;n--) b[n].o+=d;
	PMPoly *p=(PMPoly *)pm->pmpolys.a; 
	for (n=pm->pmpolys.m-1;n>=0;n--) Poly_move(&p[n].p,d);
	if (pm->irons.m)
	{
		IronRod *w=(IronRod *)pm->irons.a;
		for (n=pm->irons.m-1;n>=0;n--) w[n].c+=d;
	}
	if (pm->cuwires.m)
	{
		CurrentWire *w=(CurrentWire *)pm->cuwires.a;
		for (n=pm->cuwires.m-1;n>=0;n--) w[n].c+=d;
	}
	if (pm->cublocks.m)
	{
		CurrentBlock *b=(CurrentBlock *)pm->cublocks.a;
		for (n=pm->cublocks.m-1;n>=0;n--) b[n].o+=d;
	}
}

void Magnet_move(Magnet *pm,const REAL x,const REAL y) {Magnet_move(pm,V2_new(x,y));}

void Magnet_rotate(Magnet *pm,const REAL th)
{ // Rotates anticlockwise about the origin
	int n;
	//PMBlock *b=(PMBlock *)pm->pmblocks.a; // PMBlock don't rotate: convert to polygons?
	//for (n=pm->pmblocks.m-1;n>=0;n--) b[n].o+=d;
	PMPoly *p=(PMPoly *)pm->pmpolys.a; 
	for (n=pm->pmpolys.m-1;n>=0;n--) PMPoly_rotate(&p[n],th);
	if (pm->irons.m)
	{
		IronRod *w=(IronRod *)pm->irons.a;
		for (n=pm->irons.m-1;n>=0;n--) w[n].c=V2_rotate(w[n].c,th);
	}
	if (pm->cuwires.m)
	{
		CurrentWire *w=(CurrentWire *)pm->cuwires.a;
		for (n=pm->cuwires.m-1;n>=0;n--) w[n].c=V2_rotate(w[n].c,th);
	}
	// CurrentBlock don't rotate either
}

void Magnet_scale(Magnet *pm,const REAL k)
{
	int n; PMBlock *b=(PMBlock *)pm->pmblocks.a;
	for (n=pm->pmblocks.m-1;n>=0;n--) {b[n].o*=k; b[n].s*=k;}
	PMPoly *p=(PMPoly *)pm->pmpolys.a; 
	for (n=pm->pmpolys.m-1;n>=0;n--) Poly_scale(&p[n].p,k);
	if (pm->irons.m)
	{
		IronRod *w=(IronRod *)pm->irons.a;
		for (n=pm->irons.m-1;n>=0;n--) {w[n].c*=k; w[n].r*=k;}
	}
	if (pm->cuwires.m)
	{
		CurrentWire *w=(CurrentWire *)pm->cuwires.a;
		for (n=pm->cuwires.m-1;n>=0;n--) {w[n].c*=k; w[n].r*=k;}
	}
	if (pm->cublocks.m)
	{
		CurrentBlock *b=(CurrentBlock *)pm->cublocks.a;
		for (n=pm->cublocks.m-1;n>=0;n--) {b[n].o*=k; b[n].s*=k;}
	}
	pm->xiron*=k;
}

void Magnet_magscale(Magnet *pm,const REAL k)
{ // Scales all the magnetisation (Br) vectors
	int n; PMBlock *b=(PMBlock *)pm->pmblocks.a;
	for (n=pm->pmblocks.m-1;n>=0;n--) b[n].Br*=k;
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (n=pm->pmpolys.m-1;n>=0;n--) p[n].Br*=k;
}

void Magnet_explode(Magnet *pm,const REAL f)
{
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=pm->pmpolys.m-1;n>=0;n--) Poly_move(&p[n].p,f*Poly_centre(p[n].p));
}

V2 Magnet_PMPoly_force(const Magnet mag,const int n)
{ // Returns force on PMPoly n from the other PMPolys
	int i; V2 ret=V2_zero;
	Magnet m2=Magnet_copy(mag); Magnet_explode(&m2,1e-8);
	PMPoly *p=(PMPoly *)m2.pmpolys.a;
	for (i=m2.pmpolys.m-1;i>=0;i--) if (i!=n) ret+=PMPoly_force(p[n],p[i]);
	Magnet_free(m2);
	return ret;
}

V2 Magnet_min(const Magnet mag)
{
	PMPoly *p=(PMPoly *)mag.pmpolys.a;
	V2 ret=Poly_min(p[0].p);
	for (int n=mag.pmpolys.m-1;n>0;n--) ret=V2_min(ret,Poly_min(p[n].p));
	return ret;
}

V2 Magnet_max(const Magnet mag)
{
	PMPoly *p=(PMPoly *)mag.pmpolys.a;
	V2 ret=Poly_max(p[0].p);
	for (int n=mag.pmpolys.m-1;n>0;n--) ret=V2_max(ret,Poly_max(p[n].p));
	return ret;
}

void Magnet_recenter(Magnet *pm)
{ // Moves the centre of the bounding box to the origin
	Magnet_move(pm,-0.5*(Magnet_min(*pm)+Magnet_max(*pm)));
}

REAL Magnet_maxr(const Magnet mag)
{
	int n,i; V2 *v; REAL ret=0;
	PMPoly *p=(PMPoly *)mag.pmpolys.a;
	for (n=mag.pmpolys.m-1;n>=0;n--)
	{
		v=(V2 *)p[n].p.a;
		for (i=p[n].p.m-1;i>=0;i--) ret=Max(ret,V2_norm(v[i]));
	}
	return ret;
}

REAL Magnet_minr(const Magnet mag)
{ // Not quite right as only minimum of vertices distance
	int n,i; V2 *v; REAL ret=1e20;
	PMPoly *p=(PMPoly *)mag.pmpolys.a;
	for (n=mag.pmpolys.m-1;n>=0;n--)
	{
		v=(V2 *)p[n].p.a;
		for (i=p[n].p.m-1;i>=0;i--) ret=Min(ret,V2_norm(v[i]));
	}
	return ret;
}

REAL Magnet_maxsqr(const Magnet mag)
{ // Minimum radius of square containing magnet (centred around 0,0) 
	int n,i; V2 *v; REAL ret=0;
	PMPoly *p=(PMPoly *)mag.pmpolys.a;
	for (n=mag.pmpolys.m-1;n>=0;n--)
	{
		v=(V2 *)p[n].p.a;
		for (i=p[n].p.m-1;i>=0;i--) ret=Max(ret,Max(fabs(v[i].x),fabs(v[i].y)));
	}
	return ret;
}

REAL Magnet_maxwiremild(const Magnet mag)
{ // Returns the max (effective) diameter of the iron cuwires (actually wire 0) in mils, or zero if no iron cuwires
	REAL ret=0;
	IronRod *w=(IronRod *)mag.irons.a;
	for (int n=mag.irons.m-1;n>=0;n--)
	{
		REAL r=w[n].r*sqrt(w[n].p);
		if (r>ret) ret=r;
	}
	return (ret*2)/(1e-3*0.0254);
}

void Magnet_save(const Magnet mag,const char *filename,
	void (*ironwireqdesc)(const REAL,char *,const REAL)=NULL,const REAL length=0)
{
	int n,i,m; FILE *out=fopen(filename,"wt");
	fprintf(out,"Block name, B_rx (T),B_ry, x1 (m),y1,x2,y2,...\n");
	PMBlock *b=(PMBlock *)mag.pmblocks.a,c;
	for (n=0;n<mag.pmblocks.m;n++)
	{
		c=b[n];
		fprintf(out,"Block %d, %.9lg,%.9lg, %.9lg,%.9lg,%.9lg,%.9lg,%.9lg,%.9lg,%.9lg,%.9lg\n",
			1+n, c.Br.x,c.Br.y, c.o.x,c.o.y,c.o.x,c.o.y+c.s.y,c.o.x+c.s.x,c.o.y+c.s.y,c.o.x+c.s.x,c.o.y);
	}
	PMPoly *p=(PMPoly *)mag.pmpolys.a; V2 *v;
	for (n=0;n<mag.pmpolys.m;n++)
	{
		if (p[n].name) fprintf(out,"Poly %s, ",p[n].name); else fprintf(out,"Poly %d, ",1+n);
		fprintf(out,"%.9lg,%.9lg, ",p[n].Br.x,p[n].Br.y);
		v=(V2 *)p[n].p.a; m=p[n].p.m;
		for (i=0;i<m;i++) fprintf(out,"%.9lg,%.9lg%c",v[i].x,v[i].y,i+1==m?'\n':',');
	}
	if (mag.irons.m)
	{
		fprintf(out,"Block name, B_xhere (T),B_yhere, cx (m),cy,r, proportion, Area (mm^2),Rods required\n");
		IronRod *w=(IronRod *)mag.irons.a; REAL mm2; char str[150];
		for (n=0;n<mag.irons.m;n++)
		{
			mm2=IronRod_area(w[n])/1e-6;
			if (ironwireqdesc) ironwireqdesc(mm2,str,length); else *str=0;
			fprintf(out,"Iron wire %d, %.9lg,%.9lg, %.9lg,%.9lg,%.9lg, %.9lg, %.9lg,%s\n",1+n,
				w[n].B.x,w[n].B.y, w[n].c.x,w[n].c.y,w[n].r, w[n].p, mm2,str);
		}
	}
	if (mag.cuwires.m)
	{
		fprintf(out,"Block name, cx (m),cy,r, I (A)\n");
		CurrentWire *w=(CurrentWire *)mag.cuwires.a;
		for (n=0;n<mag.cuwires.m;n++)
			fprintf(out,"Copper wire %d, %.9lg,%.9lg,%.9lg, %.9lg\n",1+n,
				w[n].c.x,w[n].c.y,w[n].r, w[n].I);
	}
	if (mag.cublocks.m)
	{
		fprintf(out,"Block name, x1 (m),y1,x2,y2, I (A)\n");
		CurrentBlock *b=(CurrentBlock *)mag.cublocks.a;
		for (n=0;n<mag.cublocks.m;n++)
			fprintf(out,"Copper block %d, %.9lg,%.9lg,%.9lg,%.9lg, %.9lg\n",1+n,
				b[n].o.x,b[n].o.y,b[n].o.x+b[n].s.x,b[n].o.y+b[n].s.y, b[n].I);
	}
	if (mag.xiron!=0) fprintf(out,"Iron plates at X=+/-,%lg,m\n",mag.xiron);
	fclose(out);
}

Magnet Magnet_fromcsv(const csv f)
{ // In-RAM loading (in absence of proper "magnets" structure, use csv)
	int n,i,rm,m=csvm(f); Magnet ret=Magnet_new();
	for (n=0;n<m;n++)
	{
		if (striseg(f[n][0],"Block name")) continue; // Header row
		else if (striseg(f[n][0],"Block"))
		{
			PMBlock b;
			b.Br.x=atof(f[n][1]);
			b.Br.y=atof(f[n][2]);
			b.o.x=atof(f[n][3]);
			b.o.y=atof(f[n][4]);
			b.s.x=atof(f[n][7])-b.o.x;
			b.s.y=atof(f[n][6])-b.o.y;
			LSet_add(&ret.pmblocks,&b);
		}
		else if (striseg(f[n][0],"Poly "))
		{
			PMPoly p=PMPoly_new();
			char *name=f[n][0]+strlen("Poly ");
			char num[12]; sprintf(num,"%d",1+n);
			if (strcmp(name,num)) streqi(&p.name,name);
			p.Br.x=atof(f[n][1]);
			p.Br.y=atof(f[n][2]);
			rm=csvrowm(f[n]);
			for (i=4;i<rm;i+=2) Poly_addpoint(&p.p,atof(f[n][i-1]),atof(f[n][i]));
			LSet_add(&ret.pmpolys,&p);
		}
		else if (striseg(f[n][0],"Iron wire"))
		{
			IronRod w;
			w.B.x=atof(f[n][1]);
			w.B.y=atof(f[n][2]);
			w.c.x=atof(f[n][3]);
			w.c.y=atof(f[n][4]);
			w.r=atof(f[n][5]);
			w.p=atof(f[n][6]);
			LSet_add(&ret.irons,&w);
		}		
		else if (striseg(f[n][0],"Copper wire"))
		{
			CurrentWire w;
			w.c.x=atof(f[n][1]);
			w.c.y=atof(f[n][2]);
			w.r=atof(f[n][3]);
			w.I=atof(f[n][4]);
			LSet_add(&ret.cuwires,&w);
		}
		else if (striseg(f[n][0],"Copper block"))
		{
			CurrentBlock b;
			b.o.x=atof(f[n][1]);
			b.o.y=atof(f[n][2]);
			b.s.x=atof(f[n][3])-b.o.x;
			b.s.y=atof(f[n][4])-b.o.y;
			b.I=atof(f[n][5]);
			LSet_add(&ret.cublocks,&b);
		}
		else if (striseg(f[n][0],"Iron plates")) ret.xiron=atof(f[n][1]);
		else {char str[250]; sprintf(str,"Unrecognised block type in '%s'",f[n][0]); reportfatala(str);}
	}
	return ret;
}

Magnet Magnet_load(const char *filename)
{
	csv f=csvload(filename);
	if (!f) {char str[250]; sprintf(str,"Can't load file '%s' in Magnet_load",filename); reportfatala(str); return Magnet_new();}
	Magnet ret=Magnet_fromcsv(f);
	csvfree(f);
	return ret;
}

void Magnet_savevolumes(const Magnet mag,const char *filename,const REAL length=0)
{
	int n,i,m; FILE *out=fopen(filename,"wt");
	fprintf(out,"Block name,Area (cm^2),Length (cm),Volume (cm^3)\n");
	PMBlock *b=(PMBlock *)mag.pmblocks.a; double a;
	for (n=0;n<mag.pmblocks.m;n++)
	{
		a=PMBlock_area(b[n]);
		fprintf(out,"Block %d, %.9lg,%.9lg,%.9lg\n",1+n, a/1e-4,length/1e-2,a*length/1e-6);
	}
	PMPoly *p=(PMPoly *)mag.pmpolys.a;
	for (n=0;n<mag.pmpolys.m;n++)
	{
		a=Poly_area(p[n].p);
		if (p[n].name) fprintf(out,"Poly %s, ",p[n].name); else fprintf(out,"Poly %d, ",1+n);
		fprintf(out,"%.9lg,%.9lg,%.9lg\n",a/1e-4,length/1e-2,a*length/1e-6);
	}
	IronRod *w=(IronRod *)mag.irons.a;
	for (n=0;n<mag.irons.m;n++)
	{
		a=IronRod_area(w[n]);
		fprintf(out,"Iron wire %d, %.9lg,%.9lg,%.9lg\n",1+n, a/1e-4,length/1e-2,a*length/1e-6);
	}
	CurrentWire *wc=(CurrentWire *)mag.cuwires.a;
	for (n=0;n<mag.cuwires.m;n++)
	{
		a=CurrentWire_area(wc[n]);
		fprintf(out,"Copper wire %d, %.9lg,%.9lg,%.9lg\n",1+n, a/1e-4,length/1e-2,a*length/1e-6);
	}
	CurrentBlock *bc=(CurrentBlock *)mag.cublocks.a;
	for (n=0;n<mag.cublocks.m;n++)
	{
		a=CurrentBlock_area(bc[n]);
		fprintf(out,"Copper block %d, %.9lg,%.9lg,%.9lg\n",1+n, a/1e-4,length/1e-2,a*length/1e-6);
	}
	fclose(out);
}

void Magnet_savewires(const Magnet mag,const char *filename,const char *magname,
	const REAL length,const int minordiv=8,const int majordiv=16)
{	
	if (!mag.irons.m) return;
	int n; FILE *out=fopen(filename,"wt");
	IronRod *w=(IronRod *)mag.irons.a;
	fprintf(out,"Magnet %s tuned; %lgmil diameter wire; %d wires\n",
		magname,w[0].r*2/25.4e-6,mag.irons.m);
	REAL aref=M_PI*sq(w[0].r),l;
	for (n=0;n<mag.irons.m;n++)
	{
		if (n%majordiv==0) fprintf(out,"=========================\n");
		else if (n%majordiv%minordiv==0) fprintf(out,"--------------------\n");
		l=IronRod_area(w[n])/aref*length;
		fprintf(out,"Wire %d\t%.2fmm%s\n",1+n,l/1e-3,(l<2e-3?" [skip]":""));
	}
	fprintf(out,"=========================\n");
	fclose(out);
}

void Magnet_loadwires(Magnet *pm,const char *filename,const REAL length)
{
	if (!pm->irons.m) return;
	int n; FILE *in=fopen(filename,"rb");
	if (!in) {char err[300]; sprintf(err,"Could not find file '%s' in Magnet_loadwires",filename); reportfatala(err);}
	IronRod *w=(IronRod *)pm->irons.a; REAL l;
	for (char *s;s=fgetline(in);free(s))
	{
		if (!striseg(s,"Wire")) continue;
		sscanf(s,"Wire %d %lf",&n,&l);
		n--; l*=1e-3;
		if (n<0 || n>=pm->irons.m) reportfatal("Wire number out of range in Magnet_loadwires");
		w[n].p=l/length;
	}
	fclose(in);
}

void Magnet_savefieldmap(const Magnet mag,const char *filename,V2 x0,V2 x1,const V2 g,
	const int roundgrid=0)
{
	if (roundgrid)
	{
		x0.x=floor(x0.x/g.x)*g.x; x0.y=floor(x0.y/g.y)*g.y;
		x1.x=ceil(x1.x/g.x)*g.x; x1.y=ceil(x1.y/g.y)*g.y;
	}
	V2 p,B; FILE *out=fopen(filename,"wt");
	fprintf(out,"X (m),Y, Bx (T),By\n");
	for (p.x=x0.x;p.x<=x1.x+1e-9;p.x+=g.x)
		for (p.y=x0.y;p.y<=x1.y+1e-9;p.y+=g.y)
		{
			B=Magnet_field(mag,p);
			fprintf(out,"%.9lg,%.9lg, %.15lg,%.15lg\n",p.x,p.y,B.x,B.y);
		}
	fclose(out);
}

void Magnet_IronRods_init(Magnet *pm,const int IronRods_feedback)
{ // Call after geometry is updated but before fields are evaluated
	int debug=0*IronRods_feedback; FILE *out; V2 *ob; REAL b,maxa,maxdb; const REAL Bsat=2.1; // For 1008 steel
	int wirecutalg=1; // Each wire is influenced by the proportion of other wires within its Z range
	if (debug) {out=fopen("Magnet_IronRods_init.log","wt"); ob=(V2 *)malloc(pm->irons.m*sizeof(V2));}
	int n,i,maxit=(IronRods_feedback?10:1); IronRod *w=(IronRod *)pm->irons.a;
	for (n=pm->irons.m-1;n>=0;n--) w[n].B=V2_zero;
	for (i=0;i<maxit;i++)
	{
		if (debug) for (n=pm->irons.m-1;n>=0;n--) ob[n]=w[n].B;
		for (n=pm->irons.m-1;n>=0;n--)
		{
			if (wirecutalg) w[n].B=Magnet_field(*pm,w[n].c,(IronRods_feedback?n+1:0),1,w[n].p);
			else w[n].B=Magnet_field(*pm,w[n].c,(IronRods_feedback?n+1:0));
			b=V2_norm(w[n].B);
			if (b*2>Bsat) w[n].B*=Bsat/(b*2);
		}
		if (debug)
		{
			maxa=maxdb=0;
			for (n=pm->irons.m-1;n>=0;n--) {maxa=Max(maxa,V2_norm(w[n].B)); maxdb=Max(maxdb,V2_dist(ob[n],w[n].B));}
			fprintf(out,"Iteration %d->%d max B = %lg T, max delta B = %lg T\n",i,i+1,maxa,maxdb);
		}
	}
	if (debug) {fclose(out); free(ob); exit(0);}
}

V2 V2_degs(const REAL deg) {return V2_direction((90.0-deg)*M_PI/180.0);} // Degrees clockwise from 12 o'clock

#define SBCLIB_MAGNET_PM2D
#endif
