#ifndef SBCLIB_MAGNET_PM2D_SAVEPS
#include <magnet/PM2D.c>

#define PS_FONT_DEFAULT "Helvetica"
//#define PS_FONT_BOLD "Helvetica-Bold" // Doesn't work on work PC for some reason
#define PS_FONT_BOLD "NimbusSans-Bold"

void psbeginregion(FILE *out,const REAL x0,const REAL y0,const REAL x1,const REAL y1,
	const REAL scale=1)
{ // Starts file, sets bounding box with (0,0) centered on 8.5"x11" page, sets units to metres
	fprintf(out,"%%!\n");
	const REAL pt=0.0254/72,cx=0.5*8.5*72,cy=0.5*11.0*72;
	REAL s=scale/pt;
	fprintf(out,"%%%%BoundingBox: %lf %lf %lf %lf\n",cx+x0*s,cy+y0*s,cx+x1*s,cy+y1*s);
	fprintf(out,"%lf %lf translate\n",cx,cy);
	fprintf(out,"%lf %lf scale\n",s,s);
	fprintf(out,"%lf setlinewidth\n",pt);
}

void pscol(FILE *out,const unsigned c)
{
	fprintf(out,"%lf %lf %lf setrgbcolor\n",(c>>16)/255.0,((c>>8)&0xFF)/255.0,(c&0xFF)/255.0);
}

void pscolour(FILE *out,const REAL r,const REAL g,const REAL b)
{
	fprintf(out,"%lf %lf %lf setrgbcolor\n",r,g,b);
}

void psgrid(FILE *out,const REAL x0,const REAL y0,const REAL x1,const REAL y1)
{
	int n;
	pscolour(out,0.8,0.8,0.8);
	for (n=ceil(x0/1e-2);n<=floor(x1/1e-2);n++)
		fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\nstroke\n",0.01*n,y0,0.01*n,y1);
	for (n=ceil(y0/1e-2);n<=floor(y1/1e-2);n++)
		fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\nstroke\n",x0,0.01*n,x1,0.01*n);
}

void psbegingrid(FILE *out,const REAL x0,const REAL y0,const REAL x1,const REAL y1,
	const REAL scale=1)
{
	psbeginregion(out,x0,y0,x1,y1,scale);
	psgrid(out,x0,y0,x1,y1);
}

void psdotcircle(FILE *out,const REAL x,const REAL y,const REAL r)
{
	int n,sides=25; REAL th;
	for (n=0;n<sides;n++)
	{
		fprintf(out,"newpath\n");
		th=0.01+M_TWOPI*n/sides;
		fprintf(out,"%lf %lf moveto\n",x+r*cos(th),y+r*sin(th));
		th=0.01+M_TWOPI*(n+0.5)/sides;
		fprintf(out,"%lf %lf lineto\n",x+r*cos(th),y+r*sin(th));
		fprintf(out,"stroke\n");
	}
}

void psdotline(FILE *out,const REAL x1,const REAL y1,const REAL x2,const REAL y2)
{
	int n,dashes=Max(2,sqrt(sq(x2-x1)+sq(y2-y1))/5e-3+0.5);
	for (n=0;n<dashes;n++)
	{
		REAL p=(REAL)n/(dashes-0.5),q=((REAL)n+0.5)/(dashes-0.5);
		fprintf(out,"newpath\n");
		fprintf(out,"%lf %lf moveto\n",x1*(1.0-p)+x2*p,y1*(1.0-p)+y2*p);
		fprintf(out,"%lf %lf lineto\n",x1*(1.0-q)+x2*q,y1*(1.0-q)+y2*q);
		fprintf(out,"stroke\n");
	}
}

void psarrowc(FILE *out,REAL x,REAL y,const REAL dx,const REAL dy)
{ // Arrow centered on x,y (end cap scales with length)
	x-=0.5*dx, y-=0.5*dy;
	fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\nstroke\n",x,y,x+dx,y+dy);
	fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\n",x+dx*0.8+dy*0.2,y+dy*0.8-dx*0.2,x+dx,y+dy);
	fprintf(out,"%lf %lf lineto\nstroke\n",x+dx*0.8-dy*0.2,y+dy*0.8+dx*0.2);
}

void psarrow(FILE *out,const REAL x1,const REAL y1,const REAL x2,const REAL y2,const REAL cap=1.5e-3)
{
	REAL dx=x2-x1,dy=y2-y1,k=cap/sqrt(dx*dx+dy*dy); dx*=k; dy*=k;
	fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\nstroke\n",x1,y1,x2,y2);
	fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\nstroke\n",x2,y2,x2-dx+dy,y2-dy-dx);
	fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\nstroke\n",x2,y2,x2-dx-dy,y2-dy+dx);	
}

void psdoublearrow(FILE *out,const REAL x1,const REAL y1,const REAL x2,const REAL y2,const REAL cap=1.5e-3)
{
	REAL dx=x2-x1,dy=y2-y1,k=cap/sqrt(dx*dx+dy*dy); dx*=k; dy*=k;
	fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\nstroke\n",x1,y1,x2,y2);
	fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\nstroke\n",x1,y1,x1+dx+dy,y1+dy-dx);
	fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\nstroke\n",x1,y1,x1+dx-dy,y1+dy+dx);
	fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\nstroke\n",x2,y2,x2-dx+dy,y2-dy-dx);
	fprintf(out,"newpath\n%lf %lf moveto\n%lf %lf lineto\nstroke\n",x2,y2,x2-dx-dy,y2-dy+dx);	
}

void pstext(FILE *out,const REAL x,const REAL y,const char *str,const REAL pts=8,const char *fontname=PS_FONT_DEFAULT)
{
	const REAL pt=0.0254/72;
	fprintf(out,"/%s %lf selectfont\n%lf %lf moveto\n(%s) show\n",
		fontname,pts*pt,x,y,str);
}

void pstextc(FILE *out,const REAL x,const REAL y,const char *str,const REAL pts=8,const char *fontname=PS_FONT_DEFAULT)
{
	const REAL pt=0.0254/72;
	fprintf(out,"/%s %lf selectfont\n%lf (%s) stringwidth pop 0.5 mul sub %lf moveto\n(%s) show\n",
		fontname,pts*pt,x,str,y-0.375*pts*pt,str);
}

void pstextr(FILE *out,const REAL x,const REAL y,const char *str,const REAL pts=8,const char *fontname=PS_FONT_DEFAULT)
{
	const REAL pt=0.0254/72;
	fprintf(out,"/%s %lf selectfont\n%lf (%s) stringwidth pop sub %lf moveto\n(%s) show\n",
		fontname,pts*pt,x,str,y-0.375*pts*pt,str);
}

void pstextcb(FILE *out,const REAL x,const REAL y,const char *str,const REAL pts=8,const char *fontname=PS_FONT_DEFAULT)
{
	const REAL pt=0.0254/72;
	fprintf(out,"/%s %lf selectfont\n%lf (%s) stringwidth pop 0.5 mul sub %lf moveto\n(%s) show\n",
		fontname,pts*pt,x,str,y,str);
}

void psangle(FILE *out,const REAL x,const REAL y,const REAL th,const REAL length)
{ 
	const REAL pt=0.0254/72;
	fprintf(out,"%lf setlinewidth\n",0.5*pt);
	psdotline(out,x,y,x,y+length);
	REAL r=0.66*length;
	fprintf(out,"newpath\n%lf %lf %lf 90 %lf arc%s\nstroke\n",
		x,y,r,90.0-(th radians),(th>=0?"n":""));
	fprintf(out,"%lf setlinewidth\n",pt);
	char str[100];
	sprintf(str,"%.2lf",th radians);
	REAL xofs=0,yofs=0;
	if (fabs(th)<40 degrees) {xofs=0.5*(th>0?r:-r); yofs=-0.5*r;}
	pstextc(out,x+xofs+0.66*r*sin(0.5*th),y+0.66*r*cos(0.5*th),str);
}

void colmapBrf(const REAL x,const REAL y,REAL *fr,REAL *fg,REAL *fb)
{
	REAL rr=x*x+y*y,th=atan2(y,x)/M_PI*3,mn,mx;
	if (rr<1) {mx=1; mn=1.0-sqrt(rr);}
	else {mn=0; mx=1.0/sqrt(rr);}
	int q=(int)floor(th); REAL w=mn+(th-q)*(mx-mn),e=mn+mx-w;
	switch (q)
	{
		case -3: case 3: *fr=mn; *fg=e; *fb=mx; break;
		case -2: *fr=w; *fg=mn; *fb=mx; break;
		case -1: *fr=mx; *fg=mn; *fb=e; break;
		case 0: *fr=mx; *fg=w; *fb=mn; break;
		case 1: *fr=e; *fg=mx; *fb=mn; break;
		case 2: *fr=mn; *fg=mx; *fb=w; break;
	}
}

void PMPoly_writeps_block(const PMPoly p,FILE *out,const int c)
{
	int i,j; V2 *v; int vm;
	for (j=0;j<2;j++)
	{
		if (j) pscolour(out,0,0,0);
		else
		{
			if (c<0)
			{
				REAL fr,fg,fb; colmapBrf(p.Br.x,p.Br.y,&fr,&fg,&fb);
				pscolour(out,0.8*fr,0.8*fg,0.8*fb);
			}
			else pscol(out,c);
		}
		fprintf(out,"newpath\n");
		v=(V2 *)p.p.a; vm=p.p.m;
		fprintf(out,"%lf %lf moveto\n",v[0].x,v[0].y);
		for (i=1;i<vm;i++) fprintf(out,"%lf %lf lineto\n",v[i].x,v[i].y);
		fprintf(out,"closepath\n");
		if (j) fprintf(out,"stroke\n"); else fprintf(out,"fill\n");
	}
}

void PMPoly_writeps_arrow(const PMPoly p,FILE *out,const unsigned col,const REAL plot_bscale=5e-3)
{
	pscol(out,col);
	V2 c=Poly_centre(p.p);
	psarrowc(out,c.x,c.y,plot_bscale*p.Br.x,plot_bscale*p.Br.y);
}

void PMPoly_writeps_arrow_dwg(const PMPoly p,FILE *out,const unsigned col,const REAL length=2e-2)
{
	pscol(out,col);
	V2 c=Poly_centre(p.p),a=V2_normalise(p.Br)*length; REAL th=atan2(a.x,a.y);
	psarrow(out,c.x,c.y,c.x+a.x,c.y+a.y);
	psangle(out,c.x,c.y,th,length);
}

void PMPoly_writeps_label(const PMPoly p,FILE *out,const unsigned col)
{
	if (p.name)
	{
		pscol(out,col);
		V2 c=Poly_centre(p.p);
		pstextc(out,c.x,c.y,p.name,8,PS_FONT_BOLD);
	}
}

void PMPoly_writeps_label_dwg(const PMPoly p,FILE *out,const unsigned col)
{
	if (p.name)
	{
		pscol(out,col);
		const REAL mar=0.03;
		V2 b0=Poly_min(p.p),b1=Poly_max(p.p);
		char str[100]; sprintf(str,"Piece %s",p.name);
		pstext(out,b0.x-mar,b1.y+mar*1.5,str,20,PS_FONT_BOLD);
	}
}

void PMPoly_writeps_boundingbox(const PMPoly p,FILE *out,const REAL dwgscale=1.00)
{
	const REAL pt=0.0254/72,mar=0.03,sp=0.002;
	pscol(out,0);
	fprintf(out,"%lf setlinewidth\n",0.5*pt);
	V2 b0=Poly_min(p.p),b1=Poly_max(p.p),*a=(V2 *)p.p.a;
	const int top_edge=1,full_width=0;
	if (full_width)
	{
		psdotline(out,b0.x,b0.y,b0.x,b1.y+mar); // H
		psdotline(out,b1.x,b0.y,b1.x,b1.y+mar);
	}
	if (top_edge)
	{
		psdotline(out,a[0].x,b1.y,a[0].x,b1.y+mar); // Top edge [0]-[1]
		psdotline(out,a[1].x,b1.y,a[1].x,b1.y+mar);
	}
	psdotline(out,b1.x,b1.y,b0.x-mar,b1.y); // V
	psdotline(out,b1.x,b0.y,b0.x-mar,b0.y);
	fprintf(out,"%lf setlinewidth\n",pt);
	if (full_width) psdoublearrow(out,b0.x,b1.y+mar,b1.x,b1.y+mar); // H
	if (top_edge) psdoublearrow(out,a[0].x,b1.y+mar,a[1].x,b1.y+mar); // Top edge [0]-[1]
	psdoublearrow(out,b0.x-mar,b0.y,b0.x-mar,b1.y); // V
	char str[100];
	if (full_width)
	{
		sprintf(str,"%.2lf",(b1.x-b0.x)/dwgscale/1e-3); // H
		pstextcb(out,0.5*(b0.x+b1.x),b1.y+mar+sp,str);
	}
	if (top_edge)
	{
		sprintf(str,"%.2lf",(a[1].x-a[0].x)/dwgscale/1e-3); // Top edge [0]-[1]
		pstextcb(out,0.5*(a[0].x+a[1].x),b1.y+mar+sp,str);
	}
	sprintf(str,"%.2lf",(b1.y-b0.y)/dwgscale/1e-3); // V
	pstextr(out,b0.x-mar-sp,0.5*(b0.y+b1.y),str);
	pstext(out,b1.x+mar,b1.y+mar,"Lengths in mm");
	pstext(out,b1.x+mar,b1.y+mar-10.0*pt,"Angles in degrees");
	sprintf(str,"Plotted at %lg%% scale",100.0*dwgscale);
	pstext(out,b1.x+mar,b1.y+mar-20.0*pt,str);
}

void PMPoly_writeps_bottomangles(const PMPoly p,FILE *out,const REAL length=2e-2)
{
	V2 *va=(V2 *)p.p.a;
	V2 a=va[0]-va[3]; REAL th=atan2(a.x,a.y);
	psangle(out,va[3].x,va[3].y,th,length);
	a=va[1]-va[2]; th=atan2(a.x,a.y);
	psangle(out,va[2].x,va[2].y,th,length);
}

#ifndef GFX_white
#define GFX_white 0xFFFFFF
#endif

void PMPoly_writeps(const PMPoly p,FILE *out,const int c=0x808080,const REAL plot_bscale=5e-3,
	const int label=1,const int bounds=0)
{
	PMPoly_writeps_block(p,out,c);
	PMPoly_writeps_arrow(p,out,c<0?GFX_white:0x00FFFF,plot_bscale);
	if (label) PMPoly_writeps_label(p,out,c==GFX_white?0:0xFFC000);
	if (bounds) PMPoly_writeps_boundingbox(p,out);
}

void PMPoly_writeps_dwg(const PMPoly p,FILE *out,const REAL dwgscale=1.00,
	const int c=0xE0E0E0,const int label=1)
{
	PMPoly pscal=PMPoly_copy(p); Poly_scale(&pscal.p,dwgscale);
	const REAL pt=0.0254/72;
	fprintf(out,"%lf setlinewidth\n",2.0*pt);
	PMPoly_writeps_block(pscal,out,c);
	fprintf(out,"%lf setlinewidth\n",pt);
	PMPoly_writeps_arrow_dwg(pscal,out,0);
	if (label) PMPoly_writeps_label_dwg(pscal,out,0);
	PMPoly_writeps_boundingbox(pscal,out,dwgscale);
	PMPoly_writeps_bottomangles(pscal,out);
	PMPoly_free(pscal);
}

void Magnet_writeps(const Magnet mag,FILE *out,const int c=0x808080,const REAL plot_bscale=5e-3,
	const int label=1)
{ // Only PMPolys for now - have to be done in right Z order to look nice with overlaps 
	int n; PMPoly *p=(PMPoly *)mag.pmpolys.a;
	for (n=0;n<mag.pmpolys.m;n++) PMPoly_writeps_block(p[n],out,c);
	for (n=0;n<mag.pmpolys.m;n++) PMPoly_writeps_arrow(p[n],out,c<0?GFX_white:0x00FFFF,plot_bscale);
	if (label) for (n=0;n<mag.pmpolys.m;n++) PMPoly_writeps_label(p[n],out,c==GFX_white?0:0xFFC000);
}

#include <magnet/goalfield.c>

void Magnet_saveps(const Magnet mag,const char *filename,LSet *goals=NULL,const int Br_colours=0)
{
	FILE *out=fopen(filename,"wt");
	V2 b0=Magnet_min(mag),b1=Magnet_max(mag);
	psbegingrid(out,b0.x,b0.y,b1.x,b1.y);
	//fprintf(out,"/Courier %lf selectfont\n0 0 moveto\n(Magnoid) show\n",20.0*pt);
	Magnet_writeps(mag,out,Br_colours?-1:0x808080);
	if (goals)
	{
		pscol(out,0x0000FF);
		GoalField *g=(GoalField *)goals->a;
		for (int n=goals->m-1;n>=0;n--) psdotcircle(out,g[n].gfc.x,g[n].gfc.y,g[n].gfr);
	}
	fclose(out);
}

void Magnets_saveps(const Magnet *mags,const int m,const char *filename,
	const char **names=NULL,const unsigned *colours=NULL)
{
	int n; FILE *out=fopen(filename,"wt");
	REAL *ofs=(REAL *)malloc(m*sizeof(REAL));
	V2 *b0=(V2 *)malloc(m*sizeof(V2)),*b1=(V2 *)malloc(m*sizeof(V2)),
		bb0=V2_zero,bb1=V2_zero;
	for (n=0;n<m;n++)
	{
		b0[n]=Magnet_min(mags[n]); b1[n]=Magnet_max(mags[n]);
		if (n==0) ofs[n]=0;
		else ofs[n]=ofs[n-1]+1e-2*ceil((-b0[n].x+b1[n-1].x)/1e-2+1);
		bb0=V2_min(bb0,V2_new(ofs[n],0)+b0[n]);
		bb1=V2_max(bb1,V2_new(ofs[n],0)+b1[n]);
	}
	bb0-=V2_new(1e-2,1e-2); bb1+=V2_new(1e-2,1e-2);
	REAL c=floor((0.5*(bb0.x+bb1.x))/1e-2+0.5)*1e-2;
	for (n=0;n<m;n++) ofs[n]-=c; bb0.x-=c; bb1.x-=c;
	psbegingrid(out,bb0.x,bb0.y,bb1.x,bb1.y,
		Min(1,200e-3/(bb1.x-bb0.x))); // Fit in 200mm<8.5" width max
	for (n=0;n<m;n++)
	{
		Magnet mag=Magnet_copy(mags[n]);
		Magnet_move(&mag,ofs[n],0);
		Magnet_writeps(mag,out);
		Magnet_free(mag);
	}
	free(ofs); free(b0); free(b1);
	fclose(out);
}

#define SBCLIB_MAGNET_PM2D_SAVEPS
#endif
