#ifndef SBCLIB_POLY
// 2D polygon library, taken out of PM2D's polygonal magnets
#ifndef REAL
#define REAL double
#endif
#include <v2.c>
#include <lset.c>

typedef LSet Poly; // LSet(V2) with clockwise (negative) winding direction

Poly Poly_new(void) {return LSet(V2);}

Poly Poly_copy(const Poly p) {return LSet_copy(&p);}

void Poly_free(Poly *p) {LSet_free(p);}

void Poly_addpoint(Poly *pp,const REAL x,const REAL y)
	{V2 p=V2_new(x,y); LSet_add(pp,&p);}

void Poly_addpoint(Poly *pp,const V2 p) {LSet_add(pp,&p);}

void Poly_addarc(Poly *pp,const V2 c,const REAL r,const REAL thmin,const REAL thmax,const REAL dth=5 degrees)
{ // Clockwise from 12 o'clock convention
	int n,m=max(1,ceil(fabs(thmax-thmin)/dth)); REAL th; V2 p;
	for (n=0;n<=m;n++)
	{
		th=thmin+(thmax-thmin)*n/m;
		p=c+r*V2_new(sin(th),cos(th));
		Poly_addpoint(pp,p);
	}
}

Poly Poly_rect(const V2 o,const V2 s)
{
	Poly ret=Poly_new(); 
	Poly_addpoint(&ret,o.x,o.y);
	Poly_addpoint(&ret,o.x,o.y+s.y);
	Poly_addpoint(&ret,o.x+s.x,o.y+s.y);
	Poly_addpoint(&ret,o.x+s.x,o.y);
	return ret;
}

V2 Poly_min(const Poly p)
{
	V2 ret=V2_new(9e99,9e99),*v=(V2 *)p.a;
	for (int n=p.m-1;n>=0;n--) ret=V2_min(ret,v[n]);
	return ret;
}

V2 Poly_max(const Poly p)
{
	V2 ret=V2_new(-9e99,-9e99),*v=(V2 *)p.a;
	for (int n=p.m-1;n>=0;n--) ret=V2_max(ret,v[n]);
	return ret;
}

V2 Poly_bbsize(const Poly p) {return Poly_max(p)-Poly_min(p);}

REAL Poly_area(const Poly p)
{
	REAL ret=0;
	int n,m=p.m; V2 *v=(V2 *)p.a,a,b;
	for (n=m-1;n>=0;n--)
	{
		a=v[n]; b=v[(n+1)%m];
		ret+=a.y*b.x-a.x*b.y;
	}
	return 0.5*ret;
}

REAL Poly_diff(const Poly p,const Poly q)
{ // Makes some attempt to permute order of points
	if (p.m!=q.m) return abs((int)p.m-(int)q.m)*1e9;
	V2 *u=(V2 *)p.a,*v=(V2 *)q.a;
	int n,i=0,m=p.m; REAL dd,bdd=1e20;
	for (n=m-1;n>=0;n--)
	{
		dd=V2_dd(u[0],v[n]);
		if (dd<bdd) {bdd=dd; i=n;}
	}
	REAL ret=0;
	for (n=m-1;n>=0;n--) ret+=V2_dd(u[n],v[(n+i)%m]);
	return sqrt(ret);
}

int linecollide(const V2 a,const V2 b,const V2 c,const V2 d)
{
	REAL denom=(d.y-c.y)*(b.x-a.x)-(d.x-c.x)*(b.y-a.y);
	if (denom==0) return 0;
	REAL ab=((d.x-c.x)*(a.y-c.y)-(d.y-c.y)*(a.x-c.x))/denom;
	if (ab<0 || ab>1) return 0;
	REAL cd=((b.x-a.x)*(a.y-c.y)-(b.y-a.y)*(a.x-c.x))/denom;
	return cd>=0 && cd<=1;
}

void linecollidef(const V2 a,const V2 b,const V2 c,const V2 d,
	REAL *pab,REAL *pcd)
{
	REAL denom=(d.y-c.y)*(b.x-a.x)-(d.x-c.x)*(b.y-a.y);
	if (denom==0) {*pab=*pcd=-1; return;}
	*pab=((d.x-c.x)*(a.y-c.y)-(d.y-c.y)*(a.x-c.x))/denom;
	*pcd=((b.x-a.x)*(a.y-c.y)-(b.y-a.y)*(a.x-c.x))/denom;
}
/*
int Poly_inside_v1(const Poly p,const V2 r)
{
	int n,m=p.m,ret=0; V2 *v=(V2 *)p.a,
		o=Poly_min(p)-V2_new(0.987654321e-3,1.02030405e-3);
	for (n=m-1;n>=0;n--) ret^=linecollide(o,r,v[n],v[(n+1)%m]);
	return ret;
}
*/
int Poly_inside(const Poly p,const V2 r)
{ // New version, split plane into x>=r.x and x<r.x, vertices are one side or other (no degenerate cases)
	int n,i,m=p.m,ret=0; V2 *v=(V2 *)p.a;
	for (n=m-1;n>=0;n--)
	{
		i=(n+1)%m;
		if ((v[n].x>=r.x)+(v[i].x>=r.x)==1) // Lines crosses from one side to other 
			ret^=((v[n].y*(v[i].x-r.x)+v[i].y*(r.x-v[n].x))/(v[i].x-v[n].x)<r.y); // Count intersections below r.y
	}
	return ret;
}

int Poly_collideoutline(const Poly p,const Poly q)
{
	V2 *u=(V2 *)p.a,*v=(V2 *)q.a;
	for (int n=p.m-1;n>=0;n--) for (int i=q.m-1;i>=0;i--)
		if (linecollide(u[n],u[(n+1)%p.m],v[i],v[(i+1)%q.m])) return 1;
	return 0;
}

int Poly_collide(const Poly p,const Poly q)
{
	V2 *u=(V2 *)p.a,*v=(V2 *)q.a;
	if (Poly_inside(p,v[0])) return 1; // Catch one entirely inside the other
	if (Poly_inside(q,u[0])) return 1;
	return Poly_collideoutline(p,q);
}

int Poly_contains(const Poly p,const Poly q)
{ // Is q entirely inside p?
	V2 *v=(V2 *)q.a;
	if (!Poly_inside(p,v[0])) return 0;
	return !Poly_collideoutline(p,q);
}

Poly Poly_clip(const Poly p,const V2 c,const V2 d)
{ // Subtracts a half-plane from c in the direction of d
	int n,i,m=p.m; V2 *v=(V2 *)p.a;
	Poly ret=Poly_new(); REAL x0,x1;
	for (n=0;n<m;n++)
	{
		i=(n+1)%m;
		x0=(v[n]-c)*d; x1=(v[i]-c)*d;
		if (x0<0) Poly_addpoint(&ret,v[n]);
		if ((x0>=0)+(x1>=0)==1) // Lines crosses from one side to other, add intersection point
			Poly_addpoint(&ret,(v[n]*x1-v[i]*x0)/(x1-x0));
	}
	return ret;
}

void Poly_xextent(const Poly p,const REAL y,REAL *px0,REAL *px1)
{ // Returns x interval at height y
	int n,i,m=p.m; V2 *v=(V2 *)p.a; REAL x,x0=1e20,x1=-1e20;
	for (n=0;n<m;n++)
	{
		i=(n+1)%m;
		if ((v[n].y>=y)+(v[i].y>=y)==1)
		{
			x=(y-v[n].y)/(v[i].y-v[n].y);
			x=x*v[i].x+(1.0-x)*v[n].x;
			if (x<x0) x0=x; if (x>x1) x1=x;
		}
	}
	*px0=x0; *px1=x1;
}

// Use "discrete coordinate" (a,-1) point of p, (-1,b) point of q, (c,d) intersection of lines on p and q
// Go around until you find an intersection or loop back (if loop back, some sort of trivial-ish case)
// Start following polygons in prescribed directions (union/intesect is both same way, subtract is different ways)
// Keep flags array(s) for detecting loop back to previously-visited point and then the cycle becomes your result
Poly Poly_csg(const Poly p,const Poly q,const int pqside,const int qrev)
{ // Returns the connected component of p-q that contains p[0], or p[min not inside q]
	int n,i; V2 *u=(V2 *)p.a,*qv=(V2 *)q.a,*v;
	if (qrev)
	{
		v=(V2 *)malloc(q.m*sizeof(V2));
		for (i=q.m-1;i>=0;i--) v[i]=qv[q.m-1-i];
	}
	else v=qv;
	typedef struct {V2 v; REAL fp,fq; int cross,used;} CrossRec;
	CrossRec **c=(CrossRec **)malloc((p.m+1)*sizeof(CrossRec *));
	REAL fp,fq; int a=-1,b=-1;
	for (n=p.m-1;n>=0;n--)
	{
		c[n]=(CrossRec *)malloc((q.m+1)*sizeof(CrossRec));
		for (i=q.m-1;i>=0;i--)
		{
			linecollidef(u[n],u[(n+1)%p.m],v[i],v[(i+1)%q.m],&fp,&fq);
			c[n][i].v=u[n]*(1.0-fp)+u[(n+1)%p.m]*fp;
			c[n][i].fp=fp;
			c[n][i].fq=fq;
			c[n][i].cross=(fp>=0 && fp<=1 && fq>=0 && fq<=1);
			if (a<0 && c[n][i].cross) {a=n; b=i;}
			c[n][i].used=0;
		}
	}
	Poly ret;
	if (a<0) // No crossings
	{
		if (pqside) // Union
		{
			if (Poly_inside(p,v[0])) ret=Poly_copy(p);
			else if (Poly_inside(q,u[0])) ret=Poly_copy(q);
			else ret=Poly_copy(p);
		}
		else if (qrev) // Difference
		{
			if (Poly_inside(p,v[0])) ret=Poly_copy(p);
			else if (Poly_inside(q,u[0])) ret=Poly_new();
			else ret=Poly_copy(p);
		}
		else // Intersect
		{
			if (Poly_inside(p,v[0])) ret=Poly_copy(q);
			else if (Poly_inside(q,u[0])) ret=Poly_copy(p);
			else ret=Poly_new();
		}
		if (qrev) free(v);
		for (n=p.m-1;n>=0;n--) free(c[n]); free(c);
		return ret;
	}
	for (n=p.m-1;n>=0;n--)
	{
		c[n][q.m].v=u[n];
		c[n][q.m].used=0;
	}
	c[p.m]=(CrossRec *)malloc((q.m+1)*sizeof(CrossRec));
	for (i=q.m-1;i>=0;i--)
	{
		c[p.m][i].v=v[i];
		c[p.m][i].used=0;
	}
	V2 aa=u[(a+1)%p.m]-u[a],bb=v[(b+1)%q.m]-v[b];
	int pq=pqside^(aa.x*bb.y-aa.y*bb.x>0); // 0 for p, 1 for q
	ret=Poly_new();
	while (!c[a][b].used)
	{
		c[a][b].used=1;
		Poly_addpoint(&ret,c[a][b].v);
		if (a<p.m && b<q.m)
		{
			if (pq) // On q, switch to p
			{
				fp=2;
				for (i=q.m-1;i>=0;i--) if (c[a][i].cross)
					if (c[a][i].fp>c[a][b].fp && c[a][i].fp<fp) {fp=c[a][i].fp; b=i;}
				if (fp==2) {a=(a+1)%p.m; b=q.m;}
			}
			else // On p, switch to q
			{
				fq=2;
				for (n=p.m-1;n>=0;n--) if (c[n][b].cross)
					if (c[n][b].fq>c[a][b].fq && c[n][b].fq<fq) {fq=c[n][b].fq; a=n;}
				if (fq==2) {b=(b+1)%q.m; a=p.m;}
			}
			pq^=1;
		}
		else if (a==p.m) // On point b of q
		{
			fq=2;
			for (n=p.m-1;n>=0;n--) if (c[n][b].cross)
				if (c[n][b].fq<fq) {fq=c[n][b].fq; a=n;}
			if (fq==2) {b=(b+1)%q.m; a=p.m;}
		}
		else // On point a of p
		{
			fp=2;
			for (i=q.m-1;i>=0;i--) if (c[a][i].cross)
				if (c[a][i].fp<fp) {fp=c[a][i].fp; b=i;}
			if (fp==2) {a=(a+1)%p.m; b=q.m;}
		}
	}
	if (qrev) free(v);
	for (n=p.m;n>=0;n--) free(c[n]); free(c);
	return ret;
}

Poly Poly_union(const Poly p,const Poly q) {return Poly_csg(p,q,1,0);}
Poly Poly_intersect(const Poly p,const Poly q) {return Poly_csg(p,q,0,0);}
Poly Poly_subtract(const Poly p,const Poly q) {return Poly_csg(p,q,0,1);}
/**/ // CSG operations, particularly subtract (and tagged subtract, highlighting whose edges are who)

// There may be a fast O(N) algorithm for clipping a polygon by a half-plane

void Poly_permute(Poly *pp,const int i)
{ // Moves point [i] to point [0] circularly
	if (i==0) return;
	int n,m=pp->m; V2 *a=(V2 *)pp->a,*na=(V2 *)malloc(m*sizeof(V2));
	for (n=m-1;n>=0;n--) na[n]=a[(i+n)%m];
	free(a); pp->a=na; pp->buffer=m;
}

void Poly_reverse(Poly *pp)
{ // Reverses winding direction
	int n,m=pp->m; V2 *a=(V2 *)pp->a,t;
	for (n=m/2-1;n>=0;n--) {t=a[n]; a[n]=a[m-1-n]; a[m-1-n]=t;} 
}

Poly Poly_removerepeats(const Poly p)
{ // Removes adjacent repeated vertices
	int n,m=p.m; V2 *a=(V2 *)p.a; Poly ret=Poly_new();
	for (n=0;n<m;n++) if (a[n]!=a[(n+1)%m]) Poly_addpoint(&ret,a[n]);
	return ret;
}

void Poly_move(Poly *pp,const V2 x)
{
	for (int n=pp->m-1;n>=0;n--) ((V2 *)pp->a)[n]+=x;	
}

void Poly_rotateright(Poly *pp,const int r)
{ // By r right angles I think
	V2 *v=(V2 *)pp->a;
	for (int n=pp->m-1;n>=0;n--) v[n]=V2_rotateright(v[n],r);
}

void Poly_rotate(Poly *pp,const REAL th,const V2 about=V2_zero)
{
	V2 *v=(V2 *)pp->a;
	for (int n=pp->m-1;n>=0;n--) v[n]=V2_rotate(v[n]-about,th)+about;
}

V2 Poly_centre(const Poly p)
{ // Just average of vertices
	V2 *v=(V2 *)p.a,ret=V2_zero;
	for (int n=p.m-1;n>=0;n--) ret+=v[n];
	return ret/(double)p.m;
}

void Poly_recentre(Poly *pp) {Poly_move(pp,-Poly_centre(*pp));}

void Poly_rotatec(Poly *pp,const REAL th)
{ // Rotates about centre
	Poly_rotate(pp,th,Poly_centre(*pp));
}

void Poly_flipx(Poly *pp)
{
	int n,m=pp->m; V2 *v=(V2 *)pp->a,u;
	for (n=m-1;n>=0;n--) v[n].x*=-1;
	for (n=m-1;n>=m/2;n--) {u=v[n]; v[n]=v[m-1-n]; v[m-1-n]=u;}
}

void Poly_flipy(Poly *pp)
{
	int n,m=pp->m; V2 *v=(V2 *)pp->a,u;
	for (n=m-1;n>=0;n--) v[n].y*=-1;
	for (n=m-1;n>=m/2;n--) {u=v[n]; v[n]=v[m-1-n]; v[m-1-n]=u;}
}

void Poly_scale(Poly *pp,const REAL k)
{
	V2 *v=(V2 *)pp->a;
	for (int n=pp->m-1;n>=0;n--) v[n]*=k;
}

void Poly_scalexy(Poly *pp,const REAL kx,const REAL ky)
{ // k>0 to preserve winding
	V2 *v=(V2 *)pp->a;
	for (int n=pp->m-1;n>=0;n--) {v[n].x*=kx; v[n].y*=ky;}
}

void Poly_enlarge(Poly *pp,const REAL r)
{
	int n,i,m=pp->m; V2 *a=(V2 *)pp->a,u,v,d; REAL th;
	V2 *b=(V2 *)malloc(m*sizeof(V2));
	for (n=m-1;n>=0;n--)
	{
		for (i=(n-1+m)%m;a[i]==a[n];i=(i-1+m)%m); // Skip any coincident points
		u=V2_normalise(a[n]-a[i]);
		for (i=(n+1)%m;a[i]==a[n];i=(i+1)%m);
		v=V2_normalise(a[i]-a[n]);
		d=V2_normalise(u+v);
		th=acos(u*v);
		b[n]=a[n]+V2_rotateright(d,-1)*(r/cos(0.5*th));
	}
	memcpy(a,b,m*sizeof(V2)); free(b);
}

void Poly_saveCSV(const Poly p,const char *filename)
{	
	int n; FILE *out=fopen(filename,"wt");
	fprintf(out,"x,y\n");
	V2 *v=(V2 *)p.a;
	for (n=0;n<p.m;n++) fprintf(out,"%.16lg,%.16lg\n",v[n].x,v[n].y);
	fclose(out);
}

#ifdef SBCLIB_GRAPHICS
void Poly_plot(const Poly p)
{
	glBegin(GL_LINE_LOOP);
	V2 *v=(V2 *)p.a;
	for (int n=p.m-1;n>=0;n--) glVertex2f(v[n].x,v[n].y);
	glEnd();
	glColor3f(1,1,1);
	glBegin(GL_POINTS);
	for (int n=p.m-1;n>=0;n--) glVertex2f(v[n].x,v[n].y);
	glEnd();
}
#endif

void Poly_saveDXF(const Poly p,const char *filename)
{	
	int n; FILE *out=fopen(filename,"wt");
	fprintf(out,"  0\nSECTION\n  2\nENTITIES\n");
	V2 *v=(V2 *)p.a,e,f;
	for (n=0;n<p.m;n++)
	{
		e=v[n]; f=v[(n+1)%p.m];
		fprintf(out,"  0\nLINE\n  8\n0\n\
 10\n%.9lf\n 20\n%.9lf\n 30\n%.9lf\n\
 11\n%.9lf\n 21\n%.9lf\n 31\n%.9lf\n",e.x/0.0254,e.y/0.0254,0.0,f.x/0.0254,f.y/0.0254,0.0);
	}
	fprintf(out,"  0\nENDSEC\n  0\nEOF\n");
	fclose(out);
}

#define SBCLIB_POLY
#endif
