#ifndef SBCLIB_PROFILE1
// Profiling for loops in graphical applications [Stephen Brooks 2003-14]
// profile_display requires <bitmapfonts.c>, should choose appropriate font before calling
#ifdef PROFILE
#include <lset.c>
#include <rnd/eithertime.c>
#include <rnd/streqi.c>

LSet profile_strings;

char *profile_makestaticstring(const char *s)
{ // Turn a temporary string into one at a permanent address
	static char init=0;
	if (!init) {profile_strings=LSet(char *); init=1;}
	char **a=(char **)profile_strings.a;
	for (int n=profile_strings.m-1;n>=0;n--) if (!strcmp(s,a[n])) return a[n];
	char *ns; streqi(&ns,s);
	LSet_add(&profile_strings,&ns);
	return ns;
}

typedef struct {double t; const char *s; unsigned c;} ProfileNode;
const int profile_lset_m=100;
LSet profile_lset[profile_lset_m]; int profile_lset_n=0;

void profilec(const char *s,const unsigned c)
{ // Call with s=NULL to clear
	static char init=0;
	if (!init)
	{
		for (int n=profile_lset_m-1;n>=0;n--) profile_lset[n]=LSet_newalloc(sizeof(ProfileNode),20);
		init=1;
	}
	if (s)
	{
		ProfileNode n; n.t=eithertime(); n.s=s; n.c=c;
		LSet_add(&profile_lset[profile_lset_n],&n);
	}
	else LSet_clear(&profile_lset[profile_lset_n]);
}

#define profile(S) profilec(S,0)

void profile_timeprint(char *str,const double t)
{
	if (t<1e-6) sprintf(str,"%.3lg ns",t*1e9);
	else if (t<1e-3) sprintf(str,"%.3lg s",t*1e6);
	else if (t<1) sprintf(str,"%.3lg ms",t*1e3);
	else sprintf(str,"%.3lg sec",t);
}

void profile_differentiate(void)
{ // Takes differences of absolute times that are recorded initially (and removes final stopper entry)
	int n,h=profile_lset[profile_lset_n].m-1; ProfileNode *a=(ProfileNode *)profile_lset[profile_lset_n].a;
	if (h<0) return;
	for (n=0;n<h;n++) a[n].t=a[n+1].t-a[n].t;
	LSet_remove(&profile_lset[profile_lset_n],h);
}

void profile_mergesame(void)
{ // Adds up profiling segments with identical names
	int n,i; ProfileNode *a=(ProfileNode *)profile_lset[profile_lset_n].a;
	LSet ns=LSet(ProfileNode);
	for (n=0;n<profile_lset[profile_lset_n].m;n++)
	{
		for (i=ns.m-1;i>=0;i--) if (!strcmp(a[n].s,((ProfileNode *)ns.a)[i].s))
			{((ProfileNode *)ns.a)[i].t+=a[n].t; i=-10;}
		if (i>-10) LSet_add(&ns,a+n);
	}
	LSet_free(&profile_lset[profile_lset_n]); profile_lset[profile_lset_n]=ns;
}

#ifdef SBCLIB_BITMAPFONTS
int profile_usegreek=0;
void profile_timeprintbf(char *str,const double t)
{
	if (t<1e-6) sprintf(str,"%.3lg ns",t*1e9);
	else if (t<1e-3) {sprintf(str,(profile_usegreek?"%.3lg mus":"%.3lg us"),t*1e6); if (profile_usegreek) greekencode(str);}
	else if (t<1) sprintf(str,"%.3lg ms",t*1e3);
	else sprintf(str,"%.3lg sec",t);
}

#define PROFILE_MERGESAME 1
#define PROFILE_STACKED 2
#define PROFILE_HISTORY 4
// PROFILE_TIMELINE for absolute times?

void profile_display(const int ox,const int oy,const int flags)
{ // ortho_on(); glDisable(GL_LIGHTING); may be necessary
	profile(""); // Finish final segment
	int n,i,j,h=profile_lset[profile_lset_n].m-1; ProfileNode *a=(ProfileNode *)profile_lset[profile_lset_n].a;
	double t0=a[0].t,tt=a[h].t-t0,fps=1.0/(a[h].t-a[0].t),
		q=pow(10,ceil(log(tt)/log(10))),t;
	if (q/tt>=5) q/=5; else if (q/tt>=2) q/=2;
	profile_differentiate();
	if (flags&PROFILE_MERGESAME) profile_mergesame();
	h=profile_lset[profile_lset_n].m-1; a=(ProfileNode *)profile_lset[profile_lset_n].a;
	int py,x0=ox,x1=0,x2;
	for (n=h;n>=0;n--) x1=Maxi(x1,bflength(a[n].s));
	x1+=x0+BF_height; x2=x1+BF_height*5;
	q=(double)(xres-x2)/q;
	char str[50]; unsigned c;
	if (flags&PROFILE_STACKED)
	{
		t=0; int px;
		for (n=0;n<=h;n++)
		{
			tt=t+a[n].t; c=(a[n].c?a[n].c:(n/7%2?0xC0:0x80)*(0x1*(n%7%2)+0x100*(n%7/2%2)+0x10000*(n%7/4)));
			rect(px=(int)(x0+t*q),oy,(tt-t)*q,BF_height*2,c); px++;
			profile_timeprintbf(str,a[n].t);
			bfwrite(px,oy,a[n].s,GFX_white);
			bfwrite(px,oy+BF_height,str,GFX_white);
			t=tt;
		}
	}
	else for (n=h;n>=0;n--)
	{
		t=a[n].t; c=(a[n].c?a[n].c:0x00FFFF);
		profile_timeprintbf(str,t);
		bfwrite(x0,oy+n*BF_height,a[n].s,GFX_white);
		bfwrite(x1,oy+n*BF_height,str,GFX_white);
		if (flags&PROFILE_HISTORY) for (i=0;i<BF_height && i<profile_lset_m;i++)
		{
			j=(profile_lset_n-i+profile_lset_m)%profile_lset_m;
			if (n<profile_lset[j].m)
			{
				putpix(x2,py=(int)(oy+n*BF_height+i),RGB666(0,0,5));
				line(x2,py,x2+((ProfileNode *)profile_lset[j].a)[n].t*q,py,c);
			}
		}
		else
		{
			putpix(x2,py=(int)(oy+(n+0.5)*BF_height),RGB666(0,0,5));
			line(x2,py,x2+t*q,py,c);
		}
	}
	sprintf(str,"%.2lf fps",fps); bfwrite(x0,oy+(int)((h+1.5)*BF_height),str,GFX_white);
	profile_lset_n=(profile_lset_n+1)%profile_lset_m;
	profile(NULL);
}
#endif

void profile_save(const char *filename,const char mergesame)
{ // Save a large profile as a single file with pretty time formatting
	profile(""); // Finish final segment
	profile_differentiate();
	if (mergesame) profile_mergesame();
	FILE *out=fopen(filename,"wt");
	char str[50]; ProfileNode *a=(ProfileNode *)profile_lset[profile_lset_n].a;
	int n,i,m=profile_lset[profile_lset_n].m;
	for (n=0;n<m;n++)
	{
		profile_timeprint(str,a[n].t);
		fprintf(out,"%s",a[n].s);
		for (i=40-strlen(a[n].s);i>=0;i--) fprintf(out," ");
		fprintf(out,"%s\n",str);
	}
	fclose(out);
}

void profile_savecsvline(const char *filename,const char mergesame=0,const char retain=0)
{ // Adds a line to the CSV file, clears profile and everything
	profile("");
	profile_differentiate();
	if (mergesame) profile_mergesame();
	int n,m=profile_lset[profile_lset_n].m;
	ProfileNode *a=(ProfileNode *)profile_lset[profile_lset_n].a;
	char begin=0;
	FILE *out=fopen(filename,"rt");
	if (!out) begin=1; else fclose(out);
	out=fopen(filename,"at");
	if (begin) for (n=0;n<m;n++) fprintf(out,"%s%c",a[n].s,(n+1==m?'\n':','));
	for (n=0;n<m;n++) fprintf(out,"%.9lg%c",a[n].t,(n+1==m?'\n':','));
	fclose(out);
	if (!retain) profile(NULL);
}

void profile_writeline(FILE *out)
{ // Must be called after profile_display or profile_differentiate
	int n=(profile_lset_n-1+profile_lset_m)%profile_lset_m,
		m=profile_lset[n].m; ProfileNode *a=(ProfileNode *)profile_lset[n].a;
	for (n=0;n<m;n++) fprintf(out,"%lg%c",a[n].t,(n+1==m?'\n':'\t'));
}
#else
#define profile(A)
#define profilec(A,B)
#define profile_display(X,Y,F)
#define profile_writeline(A)
#endif

#define SBCLIB_PROFILE1
#endif
