// Magnet modifiers //

void Magnet_displace_xy_param(Magnet *pm,const REAL *a)
{
	if (array_1d_m(a)!=pm->pmpolys.m*2) reportfatal("Wrong length of parameters vector in magnets_displace_xy_param");
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=pm->pmpolys.m-1;n>=0;n--) Poly_move(&p[n].p,V2_new(a[n*2],a[n*2+1]));
}

void Magnet_displace_r_param(Magnet *pm,const REAL *a)
{
	if (array_1d_m(a)!=pm->pmpolys.m) reportfatal("Wrong length of parameters vector in magnets_displace_r_param");
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=pm->pmpolys.m-1;n>=0;n--) Poly_move(&p[n].p,a[n]*V2_normalise(Poly_centre(p[n].p)));
}

void Magnet_displace_r(Magnet *pm,const REAL a)
{
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=pm->pmpolys.m-1;n>=0;n--) Poly_move(&p[n].p,a*V2_normalise(Poly_centre(p[n].p)));
}

void Magnet_displace_fxy(Magnet *pm,const REAL fx,const REAL fy)
{ // Scale by different factors in X and Y
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=pm->pmpolys.m-1;n>=0;n--)
	{
		V2 c=Poly_centre(p[n].p);
		Poly_move(&p[n].p,V2_new(fx*c.x,fy*c.y));
	}
}

void Magnet_displace_top_half(Magnet *pm,const REAL dx,const REAL dy=0)
{
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=pm->pmpolys.m-1;n>=0;n--) if (Poly_centre(p[n].p).y>0) Poly_move(&p[n].p,V2_new(dx,dy));
}

void Magnet_gap_leftright_halves(Magnet *pm,const REAL r,const REAL gaptop,const REAL gapbot)
{ // r is radius from magnet centre to upper or lower edge of solid frame along the seam
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	REAL dx=0.25*(gaptop+gapbot),dth=0.25*(gapbot-gaptop)/r;
	for (int n=pm->pmpolys.m-1;n>=0;n--)
	{
		if (Poly_centre(p[n].p).x>0)
		{
			PMPoly_rotate(&p[n],dth);
			Poly_move(&p[n].p,V2_new(dx,0));
		}
		else
		{
			PMPoly_rotate(&p[n],-dth);
			Poly_move(&p[n].p,V2_new(-dx,0));
		}
	}	
}

void Magnet_half(Magnet *pm,const REAL thc)
{
	V2 d=V2_direction(thc);
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=pm->pmpolys.m-1;n>=0;n--) if (Poly_centre(p[n].p)*d<0)
	{
		LSet_removeshift(&pm->pmpolys,n);	
		p=(PMPoly *)pm->pmpolys.a;
	}
}

void Magnet_pieces(Magnet *pm,const int *p,const int m)
{ // Leaves just the PMPoly piece numbers in p[0..m-1]
	int n; char *f=(char *)calloc(pm->pmpolys.m,sizeof(char));
	for (n=m-1;n>=0;n--) f[p[n]]=1;
	for (n=pm->pmpolys.m-1;n>=0;n--) if (!f[n]) LSet_removeshift(&pm->pmpolys,n);	
	free(f);	
}

void Magnet_decimate_pieces(Magnet *pm,const int skip,const int rem=0)
{
	for (int n=pm->pmpolys.m-1;n>=0;n--) if (n%skip!=rem) LSet_removeshift(&pm->pmpolys,n);	
}

void Magnet_demagnetise(Magnet *pm) {Magnet_magscale(pm,0);}

void Magnet_magset(Magnet *pm,const REAL Br)
{ // Sets the magnitude of magnetisation vectors to Br
	int n; PMBlock *b=(PMBlock *)pm->pmblocks.a;
	for (n=pm->pmblocks.m-1;n>=0;n--) if (b[n].Br!=V2_zero) b[n].Br*=Br/V2_norm(b[n].Br);
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (n=pm->pmpolys.m-1;n>=0;n--) if (p[n].Br!=V2_zero) p[n].Br*=Br/V2_norm(p[n].Br);
}

void Magnet_maginvertpieces(Magnet *pm,const int start,const int end)
{ // Reverses the magnetisation in pieces pstart through pend
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=start-1;n<=end-1;n++)
	{
		//p[n].Br*=-1;
		V2 c=Poly_centre(p[n].p);
		c=V2_normalise(c);
		c=V2_rotateright(c,1);
		p[n].Br-=2*(p[n].Br*c)*c;
	}
}

void Magnet_magorthocompat(Magnet *pm,const REAL Br)
{ // Makes average magnetisation compatible with mix of orthogonal cuboids strength Br in one axis
	int n; PMPoly *p=(PMPoly *)pm->pmpolys.a; REAL x;
	for (n=pm->pmpolys.m-1;n>=0;n--)
	{
		x=fabs(p[n].Br.x)+fabs(p[n].Br.y);
		if (x>Br) p[n].Br*=Br/x;
	}	
}

void Magnet_pixellate(Magnet *pm,const REAL g,const REAL Br,const int samples)
{ // Replace a magnet with a similar grid of blocks
	int i,j; V2 x,x0=Magnet_min(*pm)-V2_new(5e-6,2e-6),x1=Magnet_max(*pm);
	LSet bs=LSet(PMBlock); PMBlock b; b.s=V2_new(g,g);
	for (x.x=x0.x;x.x<x1.x;x.x+=b.s.x) for (x.y=x0.y;x.y<x1.y;x.y+=b.s.y) 
	{
		b.o=x;
		b.Br=V2_zero;
		for (i=samples-1;i>=0;i--) for (j=samples-1;j>=0;j--)
			b.Br+=Magnet_magnetisation(*pm,b.o+V2_new(b.s.x*(0.5+i)/samples,b.s.y*(0.5+j)/samples));
		b.Br/=samples*samples;
		REAL p=frnd(Br); /**/ // Could store the residual and dither?
		if (p<fabs(b.Br.x)) b.Br=V2_new(b.Br.x<0?-Br:Br,0);
		else if (p<fabs(b.Br.x)+fabs(b.Br.y)) b.Br=V2_new(0,b.Br.y<0?-Br:Br);
		else b.Br=V2_zero;
		if (b.Br!=V2_zero) LSet_add(&bs,&b);
	}
	Magnet_clear(pm);
	LSet_merge(&pm->pmblocks,&bs);
	LSet_free(&bs);
}

void Magnet_topbottomswappieces(Magnet *pm)
{
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=pm->pmpolys.m-1;n>=0;n--)
	{
		V2 c=Poly_centre(p[n].p);
		REAL th=atan2(c.y,c.x);
		PMPoly_rotate(&p[n],-th*2);
	}	
}

REAL Magnet_displace_byforce(Magnet *pm,const REAL invk)
{ // invk ("inverse spring constant") is metres per Newton/m, returns maximum displacement
 	int n,m=pm->pmpolys.m; PMPoly *p=(PMPoly *)pm->pmpolys.a; REAL ret=0;
 	V2 *d=(V2 *)malloc(m*sizeof(V2));
	for (n=m-1;n>=0;n--)
	{
		d[n]=invk*Magnet_PMPoly_force(*pm,n); // /PMPoly_area(p[n]);
		ret=Max(ret,V2_norm(d[n]));
	}
	for (n=m-1;n>=0;n--) Poly_move(&p[n].p,d[n]);
	free(d);
	return ret;
}

void Magnet_rotate_byharmonic(Magnet *pm,const REAL amp,const int h,const REAL phase=0)
{
	int n,m=pm->pmpolys.m; PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (n=m-1;n>=0;n--)
	{
		V2 c=Poly_centre(p[n].p);
		REAL th=atan2(c.y,c.x);
		PMPoly_rotatec(&p[n],amp*sin(th*h+phase));
	}
}

void Magnet_clip(Magnet *pm,const V2 c,const V2 d)
{ // Subtracts a half-plane from c in the direction of d
	int n,m=pm->pmpolys.m; PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (n=m-1;n>=0;n--)
	{
		Poly pc=Poly_clip(p[n].p,c,d);
		Poly_free(&p[n].p); p[n].p=pc;
	}
}

void Magnet_clip_midplanes(Magnet *pm,const REAL ymidplane,const int midplanes)
{
	if (ymidplane<=0 || midplanes<=0) return;
	int n,m=pm->pmpolys.m; PMPoly *p=(PMPoly *)pm->pmpolys.a,pmp;
	for (n=m-1;n>=0;n--)
	{ // Trim pieces near midplane
		pmp=p[n];
		V2 c=Poly_centre(pmp.p); REAL ysign=(c.y>0?1:-1);
		if (!(midplanes==1 && c.x<0)) // Single midplane removed is always +X (theta=0)
		{
			Poly pc=Poly_clip(pmp.p,V2_new(0,ysign*ymidplane),V2_new(0,-ysign));
			Poly_free(&pmp.p); pmp.p=pc;
			if (midplanes==4) // 4-way symmetrical
			{
				REAL xsign=(c.x>0?1:-1);
				pc=Poly_clip(pmp.p,V2_new(xsign*ymidplane,0),V2_new(-xsign,0));
				Poly_free(&pmp.p); pmp.p=pc;
			}
		}
		p[n]=pmp;
	}
}

void Magnet_shrinkpieces(Magnet *pm,const REAL drminus)
{ // Shrinks sides by drminus, creating a gap of 2*drminus between adjacent pieces
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=pm->pmpolys.m-1;n>=0;n--) Poly_enlarge(&p[n].p,-drminus);
}

void Magnet_symm_xaxisyfield(Magnet *pm)
{ // Adds symmetric blocks other side of midplane (for normal field on midplane)
	int n,m=pm->pmpolys.m; PMPoly *p,q;
	for (n=0;n<m;n++)
	{
		p=(PMPoly *)pm->pmpolys.a;
		q=PMPoly_copy(p[n]);
		Poly_flipy(&q.p);
		q.Br.x*=-1; // Flipping Br_x cancels X field component on midplane
		LSet_add(&pm->pmpolys,&q);
	}
}

void Magnet_displace_xy_param_symm(Magnet *pm,const REAL *a,const REAL ymidplane=0)
{ // Midplane-symmetric version of Magnet_displace_xy_param
	int n,m=pm->pmpolys.m;
	if (array_1d_m(a)!=m/2*2) reportfatal("Wrong length of parameters vector in Magnet_displace_xy_param_symm2");
	for (n=m-1;n>=m/2;n--)
	{
		PMPoly_free(((PMPoly *)pm->pmpolys.a)[n]);
		LSet_remove(&pm->pmpolys,n);
	}
 	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (n=m/2-1;n>=0;n--)
	{
		Poly_move(&p[n].p,V2_new(a[n*2],a[n*2+1]));
		if (ymidplane>0)
		{
			V2 m=Poly_min(p[n].p);
			if (m.y<ymidplane) Poly_move(&p[n].p,V2_new(0,ymidplane-m.y));
		}
	}
	Magnet_symm_xaxisyfield(pm);
}

void Magnet_symm_rot180(Magnet *pm)
{ // Adds symmetric blocks rotated by 180 degrees (2-fold rotational symmetry)
	int n,m=pm->pmpolys.m; PMPoly *p,q;
	for (n=0;n<m;n++)
	{
		p=(PMPoly *)pm->pmpolys.a;
		q=PMPoly_copy(p[n]);
		PMPoly_rotate(&q,M_PI);
		LSet_add(&pm->pmpolys,&q);
	}
}

void Magnet_symm_quad(Magnet *pm)
{ // Quadrupole (or octupole) type symmetry, for normal odd fields on X axis
	Magnet_symm_xaxisyfield(pm);
	Magnet_symm_rot180(pm);
}

#include <rnd/normalvariate.c>

void Magnet_poserrors(Magnet *pm,const REAL d,const int gaussian=0)
{
	int n; PMBlock *b=(PMBlock *)pm->pmblocks.a;
	if (gaussian)
	{
		double x,y;
		for (n=pm->pmblocks.m-1;n>=0;n--) {twonormalvariates(&x,&y); b[n].o.x+=d*x; b[n].o.y+=d*y;}
	}
	else for (n=pm->pmblocks.m-1;n>=0;n--) {b[n].o.x+=frnd(d+d)-d; b[n].o.y+=frnd(d+d)-d;}
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	if (gaussian)
	{
		double x,y;
		for (n=pm->pmpolys.m-1;n>=0;n--) {twonormalvariates(&x,&y); Poly_move(&p[n].p,V2_new(d*x,d*y));}
	}
	else for (n=pm->pmpolys.m-1;n>=0;n--) Poly_move(&p[n].p,V2_new(frnd(d+d)-d,frnd(d+d)-d));	
}

void V2_relerror(V2 *pv,const REAL erelmag,const REAL erelperp,const int gaussian=0)
{
	if (gaussian)
	{
		double a,b; twonormalvariates(&a,&b);
		*pv+=(erelmag*a)**pv+(erelperp*b)*V2_new(pv->y,-pv->x);
	}
	else *pv+=(erelmag*(frnd(2)-1))**pv+(erelperp*(frnd(2)-1))*V2_new(pv->y,-pv->x); // Uniform +/- range
}

void Magnet_magerrors(Magnet *pm,const REAL erelmag,const REAL erelperp,const int gaussian=0)
{
	int n; PMBlock *b=(PMBlock *)pm->pmblocks.a;
	for (n=pm->pmblocks.m-1;n>=0;n--) V2_relerror(&b[n].Br,erelmag,erelperp,gaussian);
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (n=pm->pmpolys.m-1;n>=0;n--) V2_relerror(&p[n].Br,erelmag,erelperp,gaussian);	
}

void Magnet_blockstrengtherror(Magnet *pm,const int block,const REAL erel)
{ // Magnetisation strength error of a single block
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	p[block].Br*=1.0+erel;
}

void Magnet_blockmangerror(Magnet *pm,const int block,const REAL th)
{ // Magnetisation angle error of a single block
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	p[block].Br=V2_rotate(p[block].Br,th);
}

void Magnet_allblocksmangerror(Magnet *pm,const REAL th)
{ // Magnetisation angle error of a single block
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=pm->pmpolys.m-1;n>=0;n--) p[n].Br=V2_rotate(p[n].Br,th);
}

#include <array_2d.c>
#include <rnd/normaldist.c>

void Magnet_applyerrortable(Magnet *pm,const REAL * const *table,const int layers=1)
{ // table[block][0=deg sys, 1=deg s.d., 2=rel sys, 3=rel s.d.]
	int n,i; PMPoly *p=(PMPoly *)pm->pmpolys.a; V2 tot; REAL eangle,estrength;
	for (n=pm->pmpolys.m-1;n>=0;n--)
	{
		if (layers)
		{
			tot=V2_zero;
			for (i=layers;i>0;i--)
			{
				eangle=normaldist(table[n][0],sq(table[n][1])) degrees;
				estrength=normaldist(table[n][2],sq(table[n][3]));
				tot+=V2_rotate(p[n].Br*(1.0+estrength),eangle);
			}
			p[n].Br=tot/layers;
		}
		else // layers==0 gives the "systematic magnet"
		{
			eangle=table[n][0] degrees;
			estrength=table[n][2];
			p[n].Br=V2_rotate(p[n].Br*(1.0+estrength),eangle);
		}
	}
}

#include <rnd/dnormalvariate.c>

void V2_relerror_deterministic(V2 *pv,const REAL erelmag,const REAL erelperp,const unsigned seed,unsigned *pseq,const int gaussian=0)
{
	double a,b;
	if (gaussian)
	{
		a=dnormalvariate(seed,*pseq); (*pseq)++;
		b=dnormalvariate(seed,*pseq); (*pseq)++;
		*pv+=(erelmag*a)**pv+(erelperp*b)*V2_new(pv->y,-pv->x);
	}
	else
	{
		a=dfrnd(2,seed,*pseq)-1; (*pseq)++;
		b=dfrnd(2,seed,*pseq)-1; (*pseq)++;
		*pv+=(erelmag*a)**pv+(erelperp*b)*V2_new(pv->y,-pv->x); // Uniform +/- range
	}
}

void Magnet_magerrors_deterministic(Magnet *pm,const REAL erelmag,const REAL erelperp,const int seed,const int gaussian=0)
{
	int n; PMBlock *b=(PMBlock *)pm->pmblocks.a; unsigned seq=0;
	for (n=pm->pmblocks.m-1;n>=0;n--) V2_relerror_deterministic(&b[n].Br,erelmag,erelperp,seed,&seq,gaussian);
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (n=pm->pmpolys.m-1;n>=0;n--) V2_relerror_deterministic(&p[n].Br,erelmag,erelperp,seed,&seq,gaussian);	
}

V2 V2_error_deterministic(const REAL e,const unsigned seed,unsigned *pseq,const int gaussian=0)
{
	double a,b;
	if (gaussian)
	{
		a=dnormalvariate(seed,*pseq); (*pseq)++;
		b=dnormalvariate(seed,*pseq); (*pseq)++;
	}
	else // Uniform +/- ranges
	{
		a=dfrnd(2,seed,*pseq)-1; (*pseq)++;
		b=dfrnd(2,seed,*pseq)-1; (*pseq)++;
	}
	return e*V2_new(a,b);
}

void Magnet_poserrors_deterministic(Magnet *pm,const REAL e,const int seed,const int gaussian=0)
{
	int n; PMBlock *b=(PMBlock *)pm->pmblocks.a; unsigned seq=0;
	for (n=pm->pmblocks.m-1;n>=0;n--) b[n].o+=V2_error_deterministic(e,seed,&seq,gaussian);
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (n=pm->pmpolys.m-1;n>=0;n--) Poly_move(&p[n].p,V2_error_deterministic(e,seed,&seq,gaussian));	
}

void Magnet_ironposerrors(Magnet *pm,const REAL d)
{
	IronRod *w=(IronRod *)pm->irons.a;
	for (int n=pm->irons.m-1;n>=0;n--) {w[n].c.x+=frnd(d+d)-d; w[n].c.y+=frnd(d+d)-d;}
}

void Magnet_ironposerrors_deterministic(Magnet *pm,const REAL d,const int seed,const int gaussian=0)
{
	IronRod *w=(IronRod *)pm->irons.a; unsigned seq=0;
	for (int n=pm->irons.m-1;n>=0;n--) w[n].c+=V2_error_deterministic(d,seed,&seq,gaussian);
}

void Magnet_ironmasserrors(Magnet *pm,const REAL p)
{
	IronRod *w=(IronRod *)pm->irons.a;
	for (int n=pm->irons.m-1;n>=0;n--) w[n].p*=1.0+frnd(p+p)-p;
}

void Magnet_ironjoequantise(Magnet *pm,const REAL milsdiam=63,const int steps=20)
{
	IronRod *w=(IronRod *)pm->irons.a;
	REAL r=0.5*25.4e-3*0.001*milsdiam,maxa=M_PI*r*r,step=maxa/steps,a;
	for (int n=pm->irons.m-1;n>=0;n--)
	{
		a=M_PI*sq(w[n].r);
		a=floor(0.5+a/step)*step;
		w[n].r=sqrt(a/M_PI);
	}
}

void Magnet_mistake_ironwires(Magnet *pm,const REAL *a)
{ // Modifier, often toggled with if (KEY(VK_SPACE))
	int n,wm=pm->irons.m;
	if (array_1d_m(a)!=wm) reportfatal("Wrong length of parameters vector for adjusting IronRods in magnets_mistake_ironwires");
	IronRod *w=(IronRod *)pm->irons.a;
	//w[7-1].r=sqrt(sq(w[7-1].r)-2*sq(0.5*14e-3*0.0254)); // George's error (-2x14mil wires)
	w[7-1].r=sqrt(sq(w[7-1].r)+2*sq(0.5*14e-3*0.0254)); // Fix George's error (+2x14mil wires)
	//w[7-1].r=sqrt(sq(w[7-1].r)+sq(0.5*41e-3*0.0254)/60.0); // 1mm of 41mil
}

void Magnet_rotate_ironwires(Magnet *pm,const REAL th)
{
	IronRod *w=(IronRod *)pm->irons.a;
	for (int n=pm->irons.m-1;n>=0;n--) w[n].c=V2_rotate(w[n].c,th);
}

void Magnet_add_little_blocks(Magnet *pm,const REAL *a,const REAL Br=1.1,const REAL R=0.02,const REAL r=0.001)
{
	int n,m=array_1d_m(a)/2; REAL th,bx,by,br,k; PMPoly p;
	if (array_1d_m(a)!=m*2) reportfatal("Odd number of parameters given to Magnet_add_little_blocks");
	for (n=m-1;n>=0;n--)
	{
		//th=M_TWOPI*n/m;
		th=(n>=m/2)*180.0+30.0+120.0*(n%(m/2))/(m/2-1); th*=M_PI/180.0;
		p=PMPoly_new();
		Poly_addpoint(&p.p,-r,-r);
		Poly_addpoint(&p.p,r,-r);
		Poly_addpoint(&p.p,r,r);
		Poly_addpoint(&p.p,-r,r);
		Poly_move(&p.p,V2_new(R*cos(th),R*sin(th)));
		bx=a[n*2]; by=a[n*2+1]; br=sqrt(bx*bx+by*by);
		if (br>0) k=sin(br)/br; else k=1;
		p.Br=(Br*k)*V2_new(bx,by);
		LSet_add(&pm->pmpolys,&p);		
	}
}

void Magnet_add_trim_blocks(Magnet *pm,const REAL *a,const REAL Br,const REAL R,const REAL r)
{ // Br should be scaled by relative thickness
	int n,m=array_1d_m(a)/2; REAL th,bx,by,br,bth; PMPoly p;
	if (array_1d_m(a)!=m*2) reportfatal("Odd number of parameters given to Magnet_add_trim_blocks");
	for (n=m-1;n>=0;n--)
	{
		th=M_TWOPI*(n+0.5)/m;
		bx=a[n*2]; by=a[n*2+1]; br=sqrt(bx*bx+by*by); bth=atan2(by,bx);
		p=PMPoly_new();
		Poly_addpoint(&p.p,-r,-r);
		Poly_addpoint(&p.p,r,-r);
		Poly_addpoint(&p.p,r,r);
		Poly_addpoint(&p.p,-r,r);
		p.Br=Br*V2_new(1,0);
		PMPoly_rotate(&p,bth);
		br=(R+r)/Min(1,sqrt(br+0.01));
		Poly_move(&p.p,V2_new(br*cos(th),br*sin(th)));
		LSet_add(&pm->pmpolys,&p);		
	}
}

void Magnet_add_magic_fingers(Magnet *pm,const REAL *a,const REAL Br,
	const V2 s, // Block size
	const REAL ymin,const REAL Rmin=0)
{
	int n,i,j,m=array_1d_m(a)/2; REAL x1,x2,y1,h; PMBlock b;
	if (array_1d_m(a)!=m*2) reportfatal("Odd number of parameters given to Magnet_add_magic_fingers");
	for (n=m-1;n>=0;n--) for (i=1;i>=0;i--)
	{
		j=n*2+i;
		x1=s.x*(n-0.5*m); x2=x1+s.x;
		y1=Max(ymin,sqrt(noneg(Rmin*Rmin-sq(x1*x2<0?0:Min(fabs(x1),fabs(x2))))));
		h=sqrt(fabs(sin(a[j])))/sin(a[j]); if (fabs(h)>1e6 || isinf(h)) h=1e6;
		y1+=(fabs(h)-1)*1e-2;
		b.o=V2_new(x1,y1);
		b.s=s;
		b.Br=V2_new(0,(h<0?-Br:Br));
		if (i) b.o.y=-b.o.y-b.s.y;		
		LSet_add(&pm->pmblocks,&b);		
	}	
}

void Magnet_add_magic_fingers2(Magnet *pm,const REAL *a,const REAL Br,const REAL width,
	const REAL ymin,const REAL Rmin=0)
{ // Vary block heights instead, all as close to the aperture as possible
	int n,i,j,m=array_1d_m(a)/2; REAL x1,x2,y1,h; PMBlock b;
	if (array_1d_m(a)!=m*2) reportfatal("Odd number of parameters given to Magnet_add_magic_fingers2");
	for (n=m-1;n>=0;n--) for (i=1;i>=0;i--)
	{
		j=n*2+i;
		x1=width*(n-0.5*m); x2=x1+width;
		y1=Max(ymin,sqrt(noneg(Rmin*Rmin-sq(x1*x2<0?0:Min(fabs(x1),fabs(x2))))));
		h=fabs(a[j])*1e-3;
		b.o=V2_new(x1,y1);
		b.s=V2_new(width,h);
		b.Br=V2_new(0,(a[j]<0?-Br:Br));
		if (i) b.o.y=-b.o.y-b.s.y;		
		LSet_add(&pm->pmblocks,&b);		
	}
}

void Magnet_add_magic_fingers3(Magnet *pm,const REAL *a,const REAL Br,const REAL width,
	const REAL ymin,const REAL Rmin=0)
{ // fingers2 shared between horizontal and vertical magnetisations
	int n,i,j,m=array_1d_m(a)/4; REAL x1,x2,y1,h,s; PMBlock b;
	if (array_1d_m(a)!=m*4) reportfatal("Non-multiple-of-four parameters given to Magnet_add_magic_fingers3");
	for (n=m-1;n>=0;n--) for (i=3;i>=0;i--)
	{
		j=n*4+i;
		x1=width*(n-0.5*m); x2=x1+width;
		y1=Max(ymin,sqrt(noneg(Rmin*Rmin-sq(x1*x2<0?0:Min(fabs(x1),fabs(x2))))));
		h=fabs(a[j])*1e-3;
		b.o=V2_new(x1,y1);
		b.s=V2_new(width,h);
		s=(a[j]<0?-Br:Br);
		if (i/2) b.Br=V2_new(0.5*s,0); else b.Br=V2_new(0,s*0.5);
		if (i%2) b.o.y=-b.o.y-b.s.y;		
		LSet_add(&pm->pmblocks,&b);		
	}
}

void CurrentWires_addsymwires(LSet *ps,CurrentWire w,const int symm)
{
	REAL x=w.c.x,y=w.c.y;
	LSet_add(ps,&w);
	switch (symm)
	{
		case 1: // Normal dipole
			w.c=V2_new(x,-y); 
			LSet_add(ps,&w);
			w.I*=-1;
			w.c=V2_new(-x,y);
			LSet_add(ps,&w);
			w.c=V2_new(-x,-y); 
			LSet_add(ps,&w);
			break;
		case 2: // Skew dipole
			w.c=V2_new(-x,y); 
			LSet_add(ps,&w);
			w.I*=-1;
			w.c=V2_new(x,-y);
			LSet_add(ps,&w);
			w.c=V2_new(-x,-y); 
			LSet_add(ps,&w);			
			break;
		case 3: // Quadrupole I think
			w.c=V2_new(-x,y); 
			LSet_add(ps,&w);
			w.c=V2_new(x,-y);
			LSet_add(ps,&w);
			w.c=V2_new(-x,-y); 
			LSet_add(ps,&w);
			break;
		case 4: // Skew quadrupole I think
			w.c=V2_new(-x,-y); 
			LSet_add(ps,&w);
			w.I*=-1;
			w.c=V2_new(x,-y);
			LSet_add(ps,&w);
			w.c=V2_new(-x,y); 
			LSet_add(ps,&w);			
			break;
	}
}

void Magnet_add_conductors(Magnet *pm,const REAL *a,const int symm,const REAL asp=1)
{
	int n,m=array_1d_m(a);
	REAL rx=0.01194764+1e-3+5e-3+3e-3,ry=8e-3+3e-3,ro=21.5e-3;
	CurrentWire w; REAL r=0.001,x=-ro*asp,y=0,j=5e6;
	for (n=m-1;n>=0;n--)
	{
		y+=r*2;
		if (y+r>sqrt(ro*ro-sq(x/asp))) {x+=r*2; y=r+(fabs(x)<rx?(ry/rx)*sqrt(rx*rx-x*x):0);}
		w.c=V2_new(x,y); w.r=r; w.I=j*M_PI*r*r*sin(a[n]);
		CurrentWires_addsymwires(&pm->cuwires,w,symm);
	}
}

void Magnet_add_conductors_pos1(Magnet *pm,const REAL *a,const int symm,const REAL asp=1)
{ // Can use print_roundwires from "plotting.c" after this
	int n,m=array_1d_m(a);
	CurrentWire w; REAL r=0.001,ry=8e-3+3e-3,x=r,y=ry+r,j=5e6*sin(a[0]);
	w.r=r; w.I=j*M_PI*r*r;
	for (n=1;n<m;n++)
	{
		if (n>1 && n%9==1) {y+=r*2; x=r;}
		x+=r*(1.0+sin(a[n]));
		w.c=V2_new(x,y); 
		x+=r*2;
		CurrentWires_addsymwires(&pm->cuwires,w,symm);
	}
}

int Magnet_add_conductors_pos_clip(V2 *c,const int m,const double R,const double eps)
{
	int ret=0;
	REAL r,rx=0.01194764+1e-3+5e-3+3e-3,ry=8e-3+3e-3,ro=21.5e-3;
	for (int n=m-1;n>=0;n--)
	{
		if (c[n].x<R) {c[n].x=R+eps; ret++;}
		if (c[n].y<R) {c[n].y=R+eps; ret++;}
		r=V2_norm(c[n]);
		if (r+R>ro) {c[n]*=(ro-R-eps)/r; ret++;}
		r=sqrt(sq(c[n].x/rx)+sq(c[n].y/ry));
		if (r<1) {c[n]*=1.0/r+(R+eps)/V2_norm(c[n]); ret++;}
	}
	return ret;
}

/*#include "../../exp/wireavoid/circles_avoid.c"

void Magnet_add_conductors_pos(Magnet *pm,const REAL *a,const int symm,const REAL asp=1)
{ // Can use print_roundwires from "plotting.c" after this
	int n,m=array_1d_m(a);
	CurrentWire w; REAL r=0.001,j=-5e6*(0.6+0.4*sin(a[0]));
	w.r=r; w.I=j*M_PI*r*r;
	int wm=(m-1)/2;
	V2 *c=(V2 *)malloc(wm*sizeof(V2));
	for (n=wm-1;n>=0;n--) c[n]=0.02*V2_new(a[1+n*2],a[2+n*2]);
	circles_cliponly(c,wm,r,1e-6,Magnet_add_conductors_pos_clip);
	for (n=wm-1;n>=0;n--)
	{
		w.c=c[n];
		CurrentWires_addsymwires(&pm->cuwires,w,symm);
	}
	free(c);
}*/

void Magnet_orient_pieces(Magnet *pm,const int side,const REAL spacing=0)
{ // Turn all the wedges the right way up and centre them in X, Y>=0
	PMPoly *p=(PMPoly *)pm->pmpolys.a;
	for (int n=pm->pmpolys.m-1;n>=0;n--)
	{
		PMPoly_upright(&p[n],side);
		Poly_move(&p[n].p,V2_new(n*spacing,0));
	}
}

void Magnet_layers(Magnet *pm,const int layers)
{ // Only does pmpolys for now (not pmblocks)
	int n,i,m=pm->pmpolys.m;
	for (n=0;n<m;n++)
	{
		PMPoly *p=(PMPoly *)pm->pmpolys.a,q;
		p[n].Br/=layers;
		for (i=1;i<layers;i++)
		{
			q=PMPoly_copy(p[n]);
			LSet_add(&pm->pmpolys,&q);
			p=(PMPoly *)pm->pmpolys.a;
		}
	}	
}

// Debugging tests //

Magnet makemagnet_test(const int usepolys)
{
	Magnet ret=Magnet_new();
	PMBlock b;
	b.o=V2_zero;
	b.s=V2_new(3.0*0.0254,0.5*0.0254);
	//b.Br=V2_normalise(V2_new(mx-xres/2,yres/2-my))
	b.Br=V2_new(0,1);
	if (usepolys)
	{
		PMPoly p=PMPoly_newblock(b);
		LSet_add(&ret.pmpolys,&p);
	}
	else LSet_add(&ret.pmblocks,&b);
	return ret;
}

Magnet makemagnet_twotest(const double time,const double rot=0,const double xwire=0)
{ // time=eithertime(); xwire=0.02*mx/xres; optionally rot=M_TWOPI*mx/xres;
	Magnet ret=Magnet_new();
	double sep=0.05,rx=0.02,ry=0.04,th=time*0.25;//0.25*M_PI;
	PMPoly p=PMPoly_newblock(V2_new(-sep-rx,-ry),V2_new(rx*2,ry*2),1.07*V2_direction(rot));
	PMPoly_rotate(&p,th);
	LSet_add(&ret.pmpolys,&p);
	p=PMPoly_newblock(V2_new(sep-rx,-ry),V2_new(rx*2,ry*2),1.07*V2_direction(rot));
	PMPoly_rotate(&p,th);
	LSet_add(&ret.pmpolys,&p);
	IronRod w=IronRod_new(V2_zero,xwire);
	LSet_add(&ret.irons,&w);
	return ret;
}

#include <magnet/goalfield.c>

Magnet makemagnet_ironwirestest(const GoalField goal,const double somewiresfac=1)
{ // Don't call wires_init after this, puts the wires in a theoretical ideal field // somewiresfac=1.1*mx/xres;
	Magnet ret=Magnet_new();
	int n,wm=100; REAL r=0.3e-3,th; IronRod w;
	for (n=Mini(wm,wm*somewiresfac)-1;n>=0;n--)
	{
		th=M_TWOPI*n/wm;//+0.25*M_PI;
		w=IronRod_new(0.02*V2_direction(th),r*sqrt(0.5+0.5*cos(th*3)),100);
		w.B=goal.B(goal,w.c);
		LSet_add(&ret.irons,&w);
	}
	return ret;
}
