#ifndef SBCLIB_CSV
// Sort-of-minimal CSV format library [Stephen Brooks 2013]
// Utilises LSet and "explodesegs" etc.
#include <stdio.h>
#include <lset.c>
#include <rnd/explode.c>
#include <rnd/fgetline.c>

typedef char ***csv;

#define csvm(C) (((unsigned *)(C))[-1])
#define csvrowm(R) (((unsigned *)(R))[-1])

csv csvread(FILE *in,const char *sep=",",const char quotes=0,const char *commentprefix=NULL)
{ // Set sep to NULL for whitespace-separated columns
	LSet rs=LSet(char **); int lastblank=0;
	for (char *s,**e;s=fgetline(in);free(s)) if (!commentprefix || !strcseg(s,commentprefix))
	{
		lastblank=!*s;
		if (sep) e=explode(sep,s,quotes); else e=explodews(s,quotes);
		LSet_add(&rs,&e);
	}
	if (lastblank) {explodefree(((char ***)rs.a)[rs.m-1]); LSet_remove(&rs,rs.m-1);}
	char ***ret=(char ***)malloc(sizeof(unsigned)+rs.m*sizeof(char **));
	*(unsigned *)ret=rs.m;
	memcpy((unsigned char *)ret+sizeof(unsigned),rs.a,rs.m*sizeof(char **));
	LSet_free(&rs);
	return (csv)((unsigned char *)ret+sizeof(unsigned));
}

csv csvreadsome(FILE *in,const int start=0,const int end=-1,const int every=1,
	const char *sep=",",const char quotes=0,const char *commentprefix=NULL)
{ // Set sep to NULL for whitespace-separated columns
	LSet rs=LSet(char **); int lastblank=0,l=0;
	for (char *s,**e;s=fgetline(in);free(s)) if (!commentprefix || !strcseg(s,commentprefix))
	{
		if (l>=start && l%every==0)
		{
			lastblank=!*s;
			if (sep) e=explode(sep,s,quotes); else e=explodews(s,quotes);
			LSet_add(&rs,&e);
		}
		l++;
		if (end>=0 && l>=end) {free(s); break;}
	}
	if (lastblank) {explodefree(((char ***)rs.a)[rs.m-1]); LSet_remove(&rs,rs.m-1);}
	char ***ret=(char ***)malloc(sizeof(unsigned)+rs.m*sizeof(char **));
	*(unsigned *)ret=rs.m;
	memcpy((unsigned char *)ret+sizeof(unsigned),rs.a,rs.m*sizeof(char **));
	LSet_free(&rs);
	return (csv)((unsigned char *)ret+sizeof(unsigned));
}

csv csvload(const char *filename,const char *sep=",",const char quotes=0,const char *commentprefix=NULL)
{ // Set sep to NULL for whitespace-separated columns
	FILE *in=fopen(filename,"rb");
	if (!in) return NULL;
	csv ret=csvread(in,sep,quotes,commentprefix);
	fclose(in);
	return ret;
}

csv csvloadsome(const char *filename,const int start=0,const int end=-1,const int every=1,
	const char *sep=",",const char quotes=0,const char *commentprefix=NULL)
{ // Set sep to NULL for whitespace-separated columns
	FILE *in=fopen(filename,"rb");
	if (!in) return NULL;
	csv ret=csvreadsome(in,start,end,every,sep,quotes,commentprefix);
	fclose(in);
	return ret;
}

void csvsave(csv c,const char *filename)
{
	int n,i,m=csvm(c),rm;
	FILE *out=fopen(filename,"wt");
	for (n=0;n<m;n++)
	{
		rm=csvrowm(c[n]);
		for (i=0;i<rm;i++) fprintf(out,"%s%c",c[n][i],(i+1==rm?'\n':','));
	}
	fclose(out);
}

void csvfree(csv c)
{
	for (int n=csvm(c)-1;n>=0;n--) explodefree(c[n]);
	free((unsigned char *)c-sizeof(unsigned));
}

void csvdeleteheaderrow(csv c,const int h=1)
{ // Delete h header rows
	int n,m=csvm(c);
	for (n=0;n+h<m;n++) c[n]=c[n+h];
	csvm(c)-=h;
}

int csvcol(csv c,const char *s,const int h=0)
{ // Gets column number by name (h is header row)
	int n,m=csvrowm(c[h]);
	for (n=m-1;n>=0;n--) if (!strcmpi(c[h][n],s)) return n;
	return -1;
}

#define SBCLIB_CSV
#endif
