#ifndef SBCLIB_MAGNET_GOALFIELD
#include <lset.c>
#include <rnd/Max.c>
#include <array_1d.c>
#include <magnet/sipoles.c>

typedef struct GoalField_tag
{
	V2 (*B)(const struct GoalField_tag,const V2); // Goal field function
	V2 gfc; REAL gfr; // Good field centre and radius e.g. 10e-3
	REAL *norm,*skew; // Multipole arrays in T/m^n into fields (around gfc); [0] is dipole, [1] is quad etc.
} GoalField;

const GoalField GoalField_null={NULL,V2_zero,0,NULL,NULL};

V2 GoalField_frompoles(const GoalField goal,const V2 p)
{
	return sipoles_field(goal.norm,goal.skew,array_1d_m(goal.norm),p-goal.gfc);
}

GoalField GoalField_zero(const int pm)
{
	GoalField ret; ret.B=GoalField_frompoles; ret.gfc=V2_zero; ret.gfr=10e-3;
	ret.norm=array_1d_zeroed(pm); ret.skew=array_1d_zeroed(pm);
	return ret;
}

GoalField GoalField_linear(const REAL dip,const REAL quad,const REAL gfr)
{
	GoalField ret=GoalField_zero(2);
	ret.norm[0]=dip; ret.norm[1]=quad;
	ret.gfr=gfr;
	return ret;
}

GoalField GoalField_linear_range(const REAL dip,const REAL quad,const REAL x0,const REAL x1)
{ // dip specified at x=0 but aperture range [x0,x1]
	GoalField ret=GoalField_zero(2);
	ret.gfc=V2_new(0.5*(x0+x1),0);
	ret.norm[0]=dip+quad*ret.gfc.x; ret.norm[1]=quad;
	ret.gfr=0.5*(x1-x0);
	return ret;
}

GoalField GoalField_copy(const GoalField g)
{
	GoalField ret; ret.B=g.B; ret.gfc=g.gfc; ret.gfr=g.gfr;
	ret.norm=array_1d_copy(g.norm); ret.skew=array_1d_copy(g.skew);
	return ret;
}

void GoalField_free(GoalField goal) {array_1d_free(goal.norm); array_1d_free(goal.skew);}

GoalField GoalField_shiftgfc(const GoalField g,const V2 x)
{ // Constructor - keeps field the same in real space but gfc_out=x+gfc_in
	GoalField ret=GoalField_copy(g);
	sipoles_shift(g.norm,g.skew,array_1d_m(g.norm),x,ret.norm,ret.skew);
	ret.gfc+=x;
	return ret;
}

void GoalField_scale(GoalField *pg,const REAL k)
{ // Scales field magnitudes
	for (int n=array_1d_m(pg->norm)-1;n>=0;n--) {pg->norm[n]*=k; pg->skew[n]*=k;}
}

void GoalField_scalesize(GoalField *pg,const REAL k)
{ // Makes the physical size k times larger
	int n,m=array_1d_m(pg->norm); REAL p=1;
	for (n=0;n<m;n++) {pg->norm[n]*=p; pg->skew[n]*=p; p/=k;}
	pg->gfc*=k; pg->gfr*=fabs(k);
}

LSet GoalFields_horizontal(const GoalField g1,const REAL rv)
{ // For oval aperture etc., rv=Good beam radius vertically
	GoalField g; REAL x;
	int n,m=ceil(g1.gfr*2/(rv*2));
	if (m%2==0) m++; // Round up to nearest odd number so one circle is centered
	LSet ret=LSet(GoalField);
	for (n=0;n<m;n++)
	{ // This also appears to re-center aperture as it ignores g1.gfc
		x=(g1.gfr-rv)*(-1.0+2.0*n/(m-1));
		g=GoalField_shiftgfc(g1,V2_new(x,0));
		g.gfr=rv;
		LSet_add(&ret,&g);
	}
	return ret;
}

void GoalFields_move(LSet *gs,const V2 d)
{
	GoalField *g=(GoalField *)gs->a;
	for (int n=gs->m-1;n>=0;n--) g[n].gfc+=d;
}

void GoalFields_scale(LSet *gs,const REAL k)
{
	GoalField *g=(GoalField *)gs->a;
	for (int n=gs->m-1;n>=0;n--) GoalField_scale(&g[n],k);
}

void GoalFields_scalesize(LSet *gs,const REAL k)
{
	GoalField *g=(GoalField *)gs->a;
	for (int n=gs->m-1;n>=0;n--) GoalField_scalesize(&g[n],k);
}

void GoalFields_bb(const LSet *gs,V2 *pmin,V2 *pmax)
{
	GoalField *g=(GoalField *)gs->a;
	*pmin=V2_new(1e20,1e20); *pmax=V2_new(-1e20,-1e20);
	for (int n=gs->m-1;n>=0;n--)
	{
		*pmin=V2_min(*pmin,g[n].gfc-V2_new(g[n].gfr,g[n].gfr));
		*pmax=V2_max(*pmax,g[n].gfc+V2_new(g[n].gfr,g[n].gfr));
	}
}

void Goalfields_recenter(LSet *gs)
{
	V2 b0,b1; GoalFields_bb(gs,&b0,&b1);
	GoalFields_move(gs,-0.5*(b0+b1));
}

V2 GoalFields_field(const LSet *gs,const V2 p)
{
	GoalField *g=(GoalField *)gs->a;
	REAL bdd=1e20,dd; int n,bn=0;
	for (n=gs->m-1;n>=0;n--)
	{
		dd=V2_dd(p,g[n].gfc);
		if (dd<bdd) {bdd=dd; bn=n;}
	}
	return g[bn].B(g[bn],p);
}

void GoalFields_twosign(LSet *gs,const REAL sep)
{ // Creates two copies with opposite field sign, displaced by X=-0.5*sep and +0.5*sep
	int n,m=gs->m; V2 hsep=V2_new(0.5*sep,0);
	for (n=0;n<m;n++)
	{
		GoalField *g=(GoalField *)gs->a,gc=GoalField_copy(g[n]);
		GoalField_scale(&gc,-1);
		g[n].gfc-=hsep; gc.gfc+=hsep;
		LSet_add(gs,&gc);
	}
}

void GoalFields_printfile(const LSet *gs,const char *filename)
{
	int n,i,m=gs->m,pm; GoalField *g=(GoalField *)gs->a;
	FILE *out=fopen(filename,"wt");
	fprintf(out,"GoalFields has %d circles\n",m);
	for (n=0;n<m;n++)
	{
		pm=array_1d_m(g[n].norm);
		fprintf(out,"g[%d]: gfc=(%.9lg,%.9lg)m gfr=%.9lgm pm=%d",n,g[n].gfc.x,g[n].gfc.y,g[n].gfr,pm);
		for (i=0;i<pm;i++)
		{
			fprintf(out," (%.9lg,%.9lg)T",g[n].norm[i],g[n].skew[i]);
			if (i>0) {fprintf(out,"/m"); if (i>1) fprintf(out,"^%d",i);}
		}
		fprintf(out,"\n");
	}
	fclose(out);	
}

REAL *GoalField_coil(const GoalField goal)
{ // Constructs an array_1d interleaved coil (in Teslas)
	int pm=array_1d_m(goal.norm);
	REAL *ret=array_1d_zeroed(pm*2);
	//ret[0]=goal_norm[0]; ret[1]=goal_skdip; ret[2]=goal_norm[1]*r; // Linear field magnet
	REAL k=1;
	for (int n=0;n<pm;n++)
	{
		ret[n*2]=goal.norm[n]*k;
		ret[n*2+1]=goal.skew[n]*k;
		k*=goal.gfr;
	}
	return ret;
}

#ifdef SBCLIB_MAGNET_PM2D
REAL rmserror(const Magnet mag,const GoalField goal,V2 (* const errorfield)(V2)=NULL)
{
	V2 q,p,B; REAL tt=0; int denom=0; q.y=0;
	for (q.x=-goal.gfr;q.x<=goal.gfr;q.x+=0.1e-3)
	//for (REAL th=0;th<359;th+=2)
	{
		//p=goal.gfc+goal.gfr*V2_degs(th);
		p=goal.gfc+q;
		B=Magnet_field(mag,p)-goal.B(goal,p);
		if (errorfield) B+=errorfield(p);
		tt+=V2_normsq(B); denom++;
	}
	return sqrt(tt/denom);
}

REAL rmserrorall(const Magnet mag,const LSet goals,V2 (* const errorfield)(V2)=NULL)
{
	REAL tt=0; int denom=0; GoalField *gf=(GoalField *)goals.a;
	for (int n=goals.m-1;n>=0;n--) {tt+=sq(rmserror(mag,gf[n],errorfield)); denom++;}
	return sqrt(tt/denom);
}

REAL maxerror(const Magnet mag,const GoalField goal,V2 (* const errorfield)(V2)=NULL)
{
	V2 q,p,B; REAL ret=0; q.y=0;
	for (q.x=-goal.gfr;q.x<=goal.gfr;q.x+=0.1e-3)
	{
		p=goal.gfc+q;
		B=Magnet_field(mag,p)-goal.B(goal,p);
		if (errorfield) B+=errorfield(p);
		ret=Max(ret,V2_norm(B));
	}
	return ret;
}

REAL maxerrorall(const Magnet mag,const LSet goals,V2 (* const errorfield)(V2)=NULL)
{
	REAL ret=0; GoalField *gf=(GoalField *)goals.a;
	for (int n=goals.m-1;n>=0;n--) ret=Max(ret,maxerror(mag,gf[n],errorfield));
	return ret;
}
/*
REAL errornormcoil(const Magnet mag,const GoalField goal,const int pm=15)
{
	REAL *f=Magnet_rotatingcoil_interleaved(mag,goal.gfc,goal.gfr,pm),*g=GoalField_coil(goal),ret=0;
	for (int n=array_1d_m(f)-1;n>=0;n--) ret+=sq(f[n]-g[n]);
	array_1d_free(f); array_1d_free(g);
	return sqrt(ret);
}*/

void Magnet_GoalFields_savemidplane(const Magnet mag,const LSet *gs,const char *filetmpl)
{ // Use a set of goals to determine midplane extents (useful for oval magnets)
	// filetmpl is like "magnet_field_%smidplane.csv"
	int n; GoalField *gf=(GoalField *)gs->a;
	V2 x0=gf[0].gfc,x1=x0;
	for (n=gs->m-1;n>=0;n--)
	{
		x0=V2_min(x0,gf[n].gfc-V2_new(gf[n].gfr,gf[n].gfr));
		x1=V2_max(x1,gf[n].gfc+V2_new(gf[n].gfr,gf[n].gfr));
	}
	V2 xc=0.5*(x0+x1),xd=x1-x0;
	char file[500];
	if (xd.x>0 && xd.x>=xd.y)
	{
		sprintf(file,filetmpl,"h");
		Magnet_savefieldmap(mag,file,V2_new(x0.x,xc.y),V2_new(x1.x,xc.y),V2_new(1e-4,1e-4),1);
	}
	else if (xd.y>0)
	{
		sprintf(file,filetmpl,"v");	
		Magnet_savefieldmap(mag,file,V2_new(xc.x,x0.y),V2_new(xc.x,x1.y),V2_new(1e-4,1e-4),1);
	}
}
#endif

#define SBCLIB_MAGNET_GOALFIELD
#endif
